<?php

namespace Tests\Unit;

use App\Helpers\UnitConverter;
use InvalidArgumentException;
use PHPUnit\Framework\TestCase;

class UnitConverterTest extends TestCase
{
    public function test_convert_to_base_liquid_units()
    {
        // Test ml (base unit)
        $this->assertEquals(100, UnitConverter::convertToBase(100, 'ml'));
        
        // Test L to ml
        $this->assertEquals(1000, UnitConverter::convertToBase(1, 'L'));
        $this->assertEquals(500, UnitConverter::convertToBase(0.5, 'L'));
    }

    public function test_convert_to_base_solid_units()
    {
        // Test mg (base unit)
        $this->assertEquals(100, UnitConverter::convertToBase(100, 'mg'));
        
        // Test g to mg
        $this->assertEquals(1000, UnitConverter::convertToBase(1, 'g'));
        $this->assertEquals(500, UnitConverter::convertToBase(0.5, 'g'));
        
        // Test kg to mg
        $this->assertEquals(1000000, UnitConverter::convertToBase(1, 'kg'));
        $this->assertEquals(500000, UnitConverter::convertToBase(0.5, 'kg'));
    }

    public function test_convert_from_base_liquid_units()
    {
        // Test ml (base unit)
        $this->assertEquals(100, UnitConverter::convertFromBase(100, 'ml'));
        
        // Test ml to L
        $this->assertEquals(1, UnitConverter::convertFromBase(1000, 'L'));
        $this->assertEquals(0.5, UnitConverter::convertFromBase(500, 'L'));
    }

    public function test_convert_from_base_solid_units()
    {
        // Test mg (base unit)
        $this->assertEquals(100, UnitConverter::convertFromBase(100, 'mg'));
        
        // Test mg to g
        $this->assertEquals(1, UnitConverter::convertFromBase(1000, 'g'));
        $this->assertEquals(0.5, UnitConverter::convertFromBase(500, 'g'));
        
        // Test mg to kg
        $this->assertEquals(1, UnitConverter::convertFromBase(1000000, 'kg'));
        $this->assertEquals(0.5, UnitConverter::convertFromBase(500000, 'kg'));
    }

    public function test_convert_between_units()
    {
        // Liquid conversions
        $this->assertEquals(1000, UnitConverter::convert(1, 'L', 'ml'));
        $this->assertEquals(0.5, UnitConverter::convert(500, 'ml', 'L'));
        
        // Solid conversions
        $this->assertEquals(1000, UnitConverter::convert(1, 'g', 'mg'));
        $this->assertEquals(0.5, UnitConverter::convert(500, 'mg', 'g'));
        $this->assertEquals(1000000, UnitConverter::convert(1, 'kg', 'mg'));
        $this->assertEquals(0.000001, UnitConverter::convert(1, 'mg', 'kg'));
    }

    public function test_invalid_unit_throws_exception()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Unsupported unit: invalid');
        
        UnitConverter::convertToBase(100, 'invalid');
    }

    public function test_negative_value_throws_exception()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Value cannot be negative');
        
        UnitConverter::convertToBase(-100, 'ml');
    }

    public function test_incompatible_units_throws_exception()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Cannot convert between ml and mg - incompatible unit types');
        
        UnitConverter::convert(100, 'ml', 'mg');
    }

    public function test_is_valid_unit()
    {
        $this->assertTrue(UnitConverter::isValidUnit('ml'));
        $this->assertTrue(UnitConverter::isValidUnit('L'));
        $this->assertTrue(UnitConverter::isValidUnit('mg'));
        $this->assertTrue(UnitConverter::isValidUnit('g'));
        $this->assertTrue(UnitConverter::isValidUnit('kg'));
        $this->assertFalse(UnitConverter::isValidUnit('invalid'));
    }

    public function test_are_units_compatible()
    {
        // Liquid units should be compatible
        $this->assertTrue(UnitConverter::areUnitsCompatible('ml', 'L'));
        $this->assertTrue(UnitConverter::areUnitsCompatible('L', 'ml'));
        
        // Solid units should be compatible
        $this->assertTrue(UnitConverter::areUnitsCompatible('mg', 'g'));
        $this->assertTrue(UnitConverter::areUnitsCompatible('g', 'kg'));
        $this->assertTrue(UnitConverter::areUnitsCompatible('mg', 'kg'));
        
        // Liquid and solid should not be compatible
        $this->assertFalse(UnitConverter::areUnitsCompatible('ml', 'mg'));
        $this->assertFalse(UnitConverter::areUnitsCompatible('L', 'g'));
        $this->assertFalse(UnitConverter::areUnitsCompatible('ml', 'kg'));
        
        // Invalid units should not be compatible
        $this->assertFalse(UnitConverter::areUnitsCompatible('invalid', 'ml'));
        $this->assertFalse(UnitConverter::areUnitsCompatible('ml', 'invalid'));
    }

    public function test_get_base_unit()
    {
        $this->assertEquals('ml', UnitConverter::getBaseUnit('ml'));
        $this->assertEquals('ml', UnitConverter::getBaseUnit('L'));
        $this->assertEquals('mg', UnitConverter::getBaseUnit('mg'));
        $this->assertEquals('mg', UnitConverter::getBaseUnit('g'));
        $this->assertEquals('mg', UnitConverter::getBaseUnit('kg'));
    }

    public function test_get_unit_type()
    {
        $this->assertEquals('liquid', UnitConverter::getUnitType('ml'));
        $this->assertEquals('liquid', UnitConverter::getUnitType('L'));
        $this->assertEquals('solid', UnitConverter::getUnitType('mg'));
        $this->assertEquals('solid', UnitConverter::getUnitType('g'));
        $this->assertEquals('solid', UnitConverter::getUnitType('kg'));
    }

    public function test_get_supported_units()
    {
        $units = UnitConverter::getSupportedUnits();
        $this->assertContains('ml', $units);
        $this->assertContains('L', $units);
        $this->assertContains('mg', $units);
        $this->assertContains('g', $units);
        $this->assertContains('kg', $units);
        $this->assertCount(5, $units);
    }

    public function test_get_units_by_type()
    {
        $liquidUnits = UnitConverter::getUnitsByType('liquid');
        $this->assertContains('ml', $liquidUnits);
        $this->assertContains('L', $liquidUnits);
        $this->assertCount(2, $liquidUnits);
        
        $solidUnits = UnitConverter::getUnitsByType('solid');
        $this->assertContains('mg', $solidUnits);
        $this->assertContains('g', $solidUnits);
        $this->assertContains('kg', $solidUnits);
        $this->assertCount(3, $solidUnits);
    }

    public function test_get_units_by_invalid_type_throws_exception()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage("Invalid unit type: invalid. Must be 'liquid' or 'solid'");
        
        UnitConverter::getUnitsByType('invalid');
    }

    public function test_format_with_unit()
    {
        $this->assertEquals('100.00 ml', UnitConverter::formatWithUnit(100, 'ml'));
        $this->assertEquals('1.50 L', UnitConverter::formatWithUnit(1.5, 'L'));
        $this->assertEquals('500.0 mg', UnitConverter::formatWithUnit(500, 'mg', 1));
        $this->assertEquals('2 g', UnitConverter::formatWithUnit(2, 'g', 0));
    }

    public function test_get_best_display_unit()
    {
        // Test liquid units
        $result = UnitConverter::getBestDisplayUnit(500, 'ml');
        $this->assertEquals(500, $result['value']);
        $this->assertEquals('ml', $result['unit']);
        
        $result = UnitConverter::getBestDisplayUnit(1500, 'ml');
        $this->assertEquals(1.5, $result['value']);
        $this->assertEquals('L', $result['unit']);
        
        // Test solid units
        $result = UnitConverter::getBestDisplayUnit(500, 'mg');
        $this->assertEquals(500, $result['value']);
        $this->assertEquals('mg', $result['unit']);
        
        $result = UnitConverter::getBestDisplayUnit(1500, 'mg');
        $this->assertEquals(1.5, $result['value']);
        $this->assertEquals('g', $result['unit']);
        
        $result = UnitConverter::getBestDisplayUnit(1500000, 'mg');
        $this->assertEquals(1.5, $result['value']);
        $this->assertEquals('kg', $result['unit']);
    }

    public function test_format_best_unit()
    {
        $this->assertEquals('500.00 ml', UnitConverter::formatBestUnit(500, 'ml'));
        $this->assertEquals('1.50 L', UnitConverter::formatBestUnit(1500, 'ml'));
        $this->assertEquals('500.00 mg', UnitConverter::formatBestUnit(500, 'mg'));
        $this->assertEquals('1.50 g', UnitConverter::formatBestUnit(1500, 'mg'));
        $this->assertEquals('1.50 kg', UnitConverter::formatBestUnit(1500000, 'mg'));
    }

    public function test_get_best_display_unit_invalid_base_throws_exception()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage("Base unit must be 'ml' or 'mg'");
        
        UnitConverter::getBestDisplayUnit(100, 'L');
    }

    public function test_requirement_6_4_example()
    {
        // Test the specific example from requirement 6.4: 0.5L → 500ml
        $result = UnitConverter::convert(0.5, 'L', 'ml');
        $this->assertEquals(500, $result);
    }

    public function test_requirement_5_5_base_unit_storage()
    {
        // Test that we can convert to base units for storage (ml for liquids, mg for solids)
        
        // Liquids should be stored in ml
        $this->assertEquals('ml', UnitConverter::getBaseUnit('L'));
        $this->assertEquals('ml', UnitConverter::getBaseUnit('ml'));
        
        // Solids should be stored in mg
        $this->assertEquals('mg', UnitConverter::getBaseUnit('g'));
        $this->assertEquals('mg', UnitConverter::getBaseUnit('kg'));
        $this->assertEquals('mg', UnitConverter::getBaseUnit('mg'));
        
        // Test conversion to base units
        $this->assertEquals(1000, UnitConverter::convertToBase(1, 'L')); // 1L = 1000ml
        $this->assertEquals(1000, UnitConverter::convertToBase(1, 'g')); // 1g = 1000mg
        $this->assertEquals(1000000, UnitConverter::convertToBase(1, 'kg')); // 1kg = 1000000mg
    }
}