<?php

use App\Models\User;
use App\Models\Student;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Livewire\Attributes\Layout;
use Livewire\Volt\Component;

new #[Layout('layouts.guest')] class extends Component
{
    public string $name = '';
    public string $email = '';
    public string $password = '';
    public string $password_confirmation = '';
    public string $user_type = 'other_educational_institute';
    public string $otp = '';
    public ?string $verification_code = null;
    public bool $is_email_verified = false;
    public bool $show_otp_input = false;
    public bool $show_verify_email = true;
    public bool $show_verify_otp = false;

    protected array $educationalDomains = [
        '.ac.in', '.edu.in', '.gov.in', '.nic.in', '.res.in',
        '.ernet.in', '.isro.gov.in', '.drdo.in', '.nptel.iitm.ac.in',
        '.swayam.gov.in'
    ];

    public function verifyEmail()
    {
        $domain = substr($this->email, strpos($this->email, '@') + 1);

        $isEducationalDomain = false;
        foreach ($this->educationalDomains as $validDomain) {
            if (str_ends_with($domain, $validDomain)) {
                $isEducationalDomain = true;
                break;
            }
        }

        if ($this->user_type === 'other_educational_institute' && !$isEducationalDomain) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'email' => ['Email domain must be from an educational institution.']
            ]);
        } elseif ($this->user_type === 'industries' && ($isEducationalDomain || str_ends_with($domain, 'gmail.com'))) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'email' => ['Invalid email domain for industry registration.']
            ]);
        } elseif (in_array($this->user_type, ['startups', 'r_and_d_labs']) && $isEducationalDomain) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'email' => ['Invalid email domain for selected user type.']
            ]);
        }

        // Generate and send OTP
        $this->verification_code = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        session(['verification_code' => $this->verification_code]);

        // Send OTP email
        try {
            \Illuminate\Support\Facades\Mail::to($this->email)->send(new \App\Mail\OtpVerificationMail($this->verification_code));
            $this->show_otp_input = true;
            $this->show_verify_email = false;
            $this->show_verify_otp = true;
        } catch (\Exception $e) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'email' => ['Failed to send verification code. Please try again.']
            ]);
        }
    }

    public function verifyOtp()
    {
        if ($this->otp !== session('verification_code')) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'otp' => ['Invalid OTP provided.']
            ]);
        }
        session()->flash('success', 'OTP verified successfully.');

        $this->is_email_verified = true;
        $this->show_otp_input = false;
        $this->show_verify_otp = false;
    }

    /**
     * Handle an incoming registration request.
     */
    public function register(): void
    {
        $validated = $this->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'string', 'confirmed', Rules\Password::defaults()],
            'user_type' => ['required', 'string', 'in:other_educational_institute,r_and_d_labs,industries,startups'],
            'otp' => ['required', 'string', 'size:6'],
        ]);

        $validated['password'] = Hash::make($validated['password']);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => $validated['password'],
            'user_type' => $validated['user_type']
        ]);
        $user->assignRole('guest');


        // Create student entry
        Student::create([
            'user_id' => $user->id,
            'first_name' => explode(' ', $validated['name'])[0],
            'last_name' => explode(' ', $validated['name'])[1] ?? '',
            'email' => $validated['email']
        ]);

        event(new Registered($user));
        Auth::login($user);

        $this->redirect(route('dashboard', absolute: false), navigate: true);
    }
}; ?>

<div>
    <form wire:submit="register">
        @if (session()->has('success'))
            <div id="successMessage"
                 class="absolute max-w-[600px] top-4 right-4 alert alert-success bg-green-100 border-l-4 border-green-500 text-green-700 px-4 py-2 mb-4 pr-20">
                {{ session('success') }}
                <div onclick="document.getElementById('successMessage').classList.add('hidden')"
                     class="cursor-pointer bg-green-200 py-2 px-4 h-full flex justify-center absolute top-0 right-0 items-center">
                    <i class="fa fa-xmark "></i>
                </div>
            </div>
        @endif
        @if (session()->has('error'))
            <div id="dangerMessage"
                 class="absolute max-w-[600px] top-4 right-4 alert alert-danger bg-red-100 border-l-4 border-red-500 text-red-700 px-4 py-2 mb-4 pr-20">
                {{ session('error') }}
                <div onclick="document.getElementById('dangerMessage').classList.add('hidden')"
                     class="cursor-pointer bg-red-200 py-2 px-4 h-full flex justify-center absolute top-0 right-0 items-center">
                    <i class="fa fa-xmark "></i>
                </div>
            </div>
        @endif
        <div>
            <x-input-label for="user_type" :value="__('User Type')" />
            <select wire:model="user_type" id="user_type" name="user_type" {{$is_email_verified?'disabled':''}}
                    class="px-2 py-2.5 w-full text-sm font-medium bg-transparent placeholder-primary/70 border-[2px] border-primary/40 rounded-[3px] rounded-tr-[10px] rounded-bl-[10px] rounded-[4px] focus:ring-0 focus:outline-none focus:border-primary transition ease-in duration-2000"
                    required>
                <option value="other_educational_institute">Educational Institute</option>
                <option value="r_and_d_labs">R&D Labs</option>
                <option value="industries">Industries</option>
                <option value="startups">Startups</option>
            </select>
            <x-input-error :messages="$errors->get('user_type')" class="mt-2" />
        </div>

        <div class="mt-4">
            <x-input-label for="name" :value="__('Name')" />
            <x-text-input wire:model="name" id="name" class="block mt-1 w-full" placeholder="Enter your name" type="text" name="name" required autofocus autocomplete="name" />
            <x-input-error :messages="$errors->get('name')" class="mt-2" />
        </div>

        <!-- Email Address -->
        <div class="mt-4">
            <x-input-label for="email" :value="__('Email')" />
            <x-text-input wire:model="email" id="email" class="block mt-1 w-full" type="email" placeholder="Enter your email" name="email" required autocomplete="username" :disabled="$is_email_verified" />
            <x-input-error :messages="$errors->get('email')" class="mt-2" />
            @if($show_verify_email)
                <button type="button" class="w-full mt-1 px-4 py-1 text-sm bg-success/80 border-[2px] border-success/90 text-white/80 font-semibold hover:bg-success/90 hover:text-white rounded-[3px] inline-flex items-center justify-center whitespace-nowrap transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring disabled:pointer-events-none disabled:opacity-50 transition ease-in duration-200" wire:click="verifyEmail">
                    {{ __('Verify Email') }}
                </button>
            @endif
        </div>

        <!-- OTP Input -->
        @if($show_otp_input)
            <div class="mt-4">
                <x-input-label for="otp" :value="__('Enter OTP')" />
                <x-text-input wire:model="otp" id="otp" class="block mt-1 w-full" placeholder="Enter your Otp" type="text" name="otp" required />
                <x-input-error :messages="$errors->get('otp')" class="mt-2" />
                @if($show_verify_otp)
                    <button  type="button" class="w-full mt-1 px-4 py-1 text-sm bg-success/80 border-[2px] border-success/90 text-white/80 font-semibold hover:bg-success/90 hover:text-white rounded-[3px] inline-flex items-center justify-center whitespace-nowrap transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring disabled:pointer-events-none disabled:opacity-50 transition ease-in duration-200" wire:click="verifyOtp">
                        {{ __('Verify OTP') }}
                    </button>
                @endif
            </div>
        @endif

        <!-- Password -->
        <div class="mt-4">
            <x-input-label for="password" :value="__('Password')" />
            <x-text-input wire:model="password" id="password" class="block mt-1 w-full" placeholder="Enter your password"
                            type="password"
                            name="password"
                            required autocomplete="new-password" />
            <x-input-error :messages="$errors->get('password')" class="mt-2" />
        </div>

        <!-- Confirm Password -->
        <div class="mt-4">
            <x-input-label for="password_confirmation" :value="__('Confirm Password')" />
            <x-text-input wire:model="password_confirmation" id="password_confirmation" class="block mt-1 w-full" placeholder="Confirm your password"
                            type="password"
                            name="password_confirmation" required autocomplete="new-password" />
            <x-input-error :messages="$errors->get('password_confirmation')" class="mt-2" />
        </div>

        <div class="flex items-center justify-end mt-4">
            <a class="underline text-sm text-gray-600 hover:text-gray-900 rounded-md focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500" href="{{ route('login') }}" wire:navigate>
                {{ __('Already registered?') }}
            </a>

            <x-primary-button class="ms-4" :disabled="!$is_email_verified">
                {{ __('Register') }}
            </x-primary-button>
        </div>
    </form>
</div>
