<?php

namespace App\Traits;

trait InventoryValidationTrait
{
    /**
     * Validate stock quantity for transactions
     */
    public function validateStockQuantity($currentQuantity, $requestedQuantity)
    {
        if ($requestedQuantity <= 0) {
            return ['valid' => false, 'message' => 'Quantity must be greater than zero'];
        }

        if ($requestedQuantity > $currentQuantity) {
            return ['valid' => false, 'message' => 'Insufficient stock available'];
        }

        return ['valid' => true, 'message' => ''];
    }

    /**
     * Validate expiry date for stock operations
     */
    public function validateExpiryDate($expiryDate, $allowExpired = false)
    {
        if (!$expiryDate) {
            return ['valid' => true, 'message' => '', 'warning' => false];
        }

        $now = now();
        $expiry = \Carbon\Carbon::parse($expiryDate);

        if ($expiry->isPast()) {
            if (!$allowExpired) {
                return ['valid' => false, 'message' => 'Item has expired', 'warning' => false];
            }
            return ['valid' => true, 'message' => 'Item has expired - requires approval', 'warning' => true];
        }

        if ($expiry->diffInDays($now) <= 30) {
            return ['valid' => true, 'message' => 'Item expires soon', 'warning' => true];
        }

        return ['valid' => true, 'message' => '', 'warning' => false];
    }

    /**
     * Format quantity with appropriate units
     */
    public function formatQuantity($quantity, $unit = null)
    {
        if (!$unit) {
            return number_format($quantity, 2);
        }

        // Convert from base units to display units
        switch ($unit) {
            case 'ml':
                if ($quantity >= 1000) {
                    return number_format($quantity / 1000, 2) . ' L';
                }
                return number_format($quantity, 0) . ' ml';
            case 'mg':
                if ($quantity >= 1000000) {
                    return number_format($quantity / 1000000, 2) . ' kg';
                } elseif ($quantity >= 1000) {
                    return number_format($quantity / 1000, 2) . ' g';
                }
                return number_format($quantity, 0) . ' mg';
            default:
                return number_format($quantity, 2) . ' ' . $unit;
        }
    }
}