<?php

namespace App\Rules;

use Closure;
use Illuminate\Contracts\Validation\ValidationRule;
use App\Helpers\UnitConverter;

class UnitConversionRule implements ValidationRule
{
    private $fromUnit;
    private $toUnit;

    public function __construct($fromUnit, $toUnit = null)
    {
        $this->fromUnit = $fromUnit;
        $this->toUnit = $toUnit;
    }

    /**
     * Run the validation rule.
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        if (empty($value) || !is_numeric($value)) {
            return; // Let other rules handle basic validation
        }

        $quantity = floatval($value);

        if ($quantity <= 0) {
            return; // Let other rules handle this
        }

        try {
            // Validate the from unit
            if (!UnitConverter::isValidUnit($this->fromUnit)) {
                $fail("Invalid unit: {$this->fromUnit}");
                return;
            }

            // Test conversion to base unit
            UnitConverter::convertToBase($quantity, $this->fromUnit);

            // If target unit specified, test conversion
            if ($this->toUnit) {
                if (!UnitConverter::isValidUnit($this->toUnit)) {
                    $fail("Invalid target unit: {$this->toUnit}");
                    return;
                }

                if (!UnitConverter::areUnitsCompatible($this->fromUnit, $this->toUnit)) {
                    $fail("Units {$this->fromUnit} and {$this->toUnit} are not compatible");
                    return;
                }

                UnitConverter::convert($quantity, $this->fromUnit, $this->toUnit);
            }

        } catch (\InvalidArgumentException $e) {
            $fail('Unit conversion error: ' . $e->getMessage());
        } catch (\Exception $e) {
            $fail('Invalid unit conversion');
        }
    }
}