<?php

namespace App\Rules;

use Closure;
use Illuminate\Contracts\Validation\ValidationRule;
use Carbon\Carbon;

class ExpiryDateRule implements ValidationRule
{
    protected $allowPastDates;
    protected $warningDays;

    public function __construct($allowPastDates = false, $warningDays = 30)
    {
        $this->allowPastDates = $allowPastDates;
        $this->warningDays = $warningDays;
    }

    /**
     * Run the validation rule.
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        if (!$value) {
            return; // Allow null/empty expiry dates
        }

        try {
            $expiryDate = Carbon::parse($value);
            $now = now();

            if ($expiryDate->isPast() && !$this->allowPastDates) {
                $fail('The expiry date cannot be in the past.');
                return;
            }

            // For new stock intake, expiry should be reasonable (not too far in past for manufacturing)
            if ($expiryDate->diffInYears($now) > 10) {
                $fail('The expiry date seems unrealistic.');
                return;
            }

        } catch (\Exception $e) {
            $fail('The expiry date format is invalid.');
        }
    }
}