<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;

class StockTransaction extends Model
{
    protected $fillable = [
        'batch_id', 'batch_item_id', 'inventory_item_id', 'item_variant_id', 'transaction_type',
        'quantity', 'remaining_quantity', 'used_for', 'conducted_by',
        'approved_by', 'transaction_date', 'notes'
    ];
    
    protected $casts = [
        'transaction_date' => 'date',
        'transaction_type' => 'string'
    ];
    
    public function batch()
    {
        return $this->belongsTo(StockBatch::class, 'batch_id');
    }
    
    public function inventoryItem()
    {
        return $this->belongsTo(InventoryItem::class);
    }
    
    public function variant()
    {
        return $this->belongsTo(ItemVariant::class, 'item_variant_id');
    }
    
    public function batchItem()
    {
        return $this->belongsTo(BatchItem::class, 'batch_item_id');
    }
    
    public function isStockIn()
    {
        return $this->transaction_type === 'in';
    }
    
    public function isStockOut()
    {
        return $this->transaction_type === 'out';
    }
    
    public function getTransactionTypeDisplayAttribute()
    {
        return ucfirst($this->transaction_type);
    }
    
    protected static function boot()
    {
        parent::boot();
        
        // Trigger alert generation after stock transactions
        static::created(function ($transaction) {
            // Run alert generation in the background to avoid slowing down the transaction
            try {
                Artisan::call('inventory:generate-alerts');
            } catch (\Exception $e) {
                // Log error but don't fail the transaction
                \Log::error('Failed to generate alerts after stock transaction: ' . $e->getMessage());
            }
        });
    }
}
