<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class StockBatch extends Model
{
    protected $fillable = [
        'inventory_item_id', 'item_variant_id', 'batch_number', 'expiry_date',
        'manufacture_date', 'supplier_id', 'initial_quantity',
        'current_quantity', 'unit_cost', 'received_date', 'notes'
    ];
    
    protected $casts = [
        'expiry_date' => 'date',
        'manufacture_date' => 'date',
        'received_date' => 'date',
        'unit_cost' => 'decimal:2'
    ];
    
    public function inventoryItem()
    {
        return $this->belongsTo(InventoryItem::class);
    }
    
    public function variant()
    {
        return $this->belongsTo(ItemVariant::class, 'item_variant_id');
    }
    
    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }
    
    public function transactions()
    {
        return $this->hasMany(StockTransaction::class, 'batch_id');
    }
    
    public function alerts()
    {
        return $this->hasMany(StockAlert::class, 'batch_id');
    }
    
    public function batchItems()
    {
        return $this->hasMany(BatchItem::class, 'stock_batch_id');
    }
    
    public function isExpired()
    {
        return $this->expiry_date && $this->expiry_date->isPast();
    }
    
    public function isExpiringSoon($days = 30)
    {
        return $this->expiry_date && $this->expiry_date->diffInDays(now()) <= $days;
    }
    
    public function getUsagePercentageAttribute()
    {
        if ($this->initial_quantity == 0) {
            return 0;
        }
        return (($this->initial_quantity - $this->current_quantity) / $this->initial_quantity) * 100;
    }
}
