<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class StockAlert extends Model
{
    protected $fillable = [
        'inventory_item_id', 'batch_id', 'alert_type', 'alert_message',
        'resolved', 'resolved_by', 'resolved_at'
    ];
    
    protected $casts = [
        'resolved' => 'boolean',
        'resolved_at' => 'datetime',
        'alert_type' => 'string'
    ];
    
    public function inventoryItem()
    {
        return $this->belongsTo(InventoryItem::class);
    }
    
    public function batch()
    {
        return $this->belongsTo(StockBatch::class, 'batch_id');
    }
    
    public function resolve($resolvedBy)
    {
        $this->update([
            'resolved' => true,
            'resolved_by' => $resolvedBy,
            'resolved_at' => now()
        ]);
    }
    
    public function isLowStockAlert()
    {
        return $this->alert_type === 'low_stock';
    }
    
    public function isExpiryAlert()
    {
        return $this->alert_type === 'expiry';
    }
    
    public function getAlertTypeDisplayAttribute()
    {
        return ucwords(str_replace('_', ' ', $this->alert_type));
    }
    
    public function scopeUnresolved($query)
    {
        return $query->where('resolved', false);
    }
    
    public function scopeResolved($query)
    {
        return $query->where('resolved', true);
    }
    
    public function scopeLowStock($query)
    {
        return $query->where('alert_type', 'low_stock');
    }
    
    public function scopeExpiry($query)
    {
        return $query->where('alert_type', 'expiry');
    }
    
    public function scopeCritical($query)
    {
        return $query->where('resolved', false);
    }
    
    public function getFormattedCreatedAtAttribute()
    {
        return $this->created_at->format('M d, Y H:i');
    }
    
    public function getFormattedResolvedAtAttribute()
    {
        return $this->resolved_at ? $this->resolved_at->format('M d, Y H:i') : null;
    }
    
    public function getPriorityAttribute()
    {
        if ($this->alert_type === 'expiry' && $this->batch) {
            $daysUntilExpiry = now()->diffInDays($this->batch->expiry_date, false);
            if ($daysUntilExpiry <= 0) return 'critical';
            if ($daysUntilExpiry <= 7) return 'high';
            if ($daysUntilExpiry <= 14) return 'medium';
            return 'low';
        }
        
        if ($this->alert_type === 'low_stock') {
            return 'high';
        }
        
        return 'medium';
    }
    
    public function getPriorityColorAttribute()
    {
        switch ($this->priority) {
            case 'critical':
                return 'red';
            case 'high':
                return 'orange';
            case 'medium':
                return 'yellow';
            case 'low':
                return 'blue';
            default:
                return 'gray';
        }
    }
}
