<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ItemVariant extends Model
{
    protected $fillable = [
        'inventory_item_id', 'volume_per_unit', 'unit_type', 'barcode'
    ];
    
    protected $casts = [
        'unit_type' => 'string',
        'volume_per_unit' => 'decimal:2'
    ];
    
    public function inventoryItem()
    {
        return $this->belongsTo(InventoryItem::class);
    }
    
    public function stockBatches()
    {
        return $this->hasMany(StockBatch::class);
    }
    
    public function transactions()
    {
        return $this->hasMany(StockTransaction::class);
    }
    
    public function getDisplayNameAttribute()
    {
        return $this->volume_per_unit . $this->unit_type;
    }
    
    public function getFormattedDisplayAttribute()
    {
        $unitNames = [
            'ml' => 'milliliters',
            'L' => 'liters',
            'mg' => 'milligrams', 
            'g' => 'grams',
            'kg' => 'kilograms'
        ];
        
        $unitName = $unitNames[$this->unit_type] ?? $this->unit_type;
        return $this->volume_per_unit . ' ' . $unitName;
    }
    
    public function isLiquidUnit()
    {
        return in_array($this->unit_type, ['ml', 'L']);
    }
    
    public function isSolidUnit()
    {
        return in_array($this->unit_type, ['mg', 'g', 'kg']);
    }
    
    public function hasStockBatches()
    {
        return $this->stockBatches()->count() > 0;
    }
    
    public function hasTransactions()
    {
        return $this->transactions()->count() > 0;
    }
    
    public function canBeDeleted()
    {
        return !$this->hasStockBatches() && !$this->hasTransactions();
    }
    
    public function getTotalStockQuantity()
    {
        return $this->stockBatches()->sum('current_quantity');
    }
    
    public function getActiveStockBatches()
    {
        return $this->stockBatches()->where('current_quantity', '>', 0);
    }
}
