<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Instrument extends Model
{

    protected $fillable = [
        'instrument_category_id',
        'lab_id',
        'technician_id',
        'name',
        'model_number',
        'serial_number',
        'description',
        'operating_status',
        'engineer_name',
        'engineer_email',
        'engineer_mobile',
        'engineer_address',
        'booking_type',
        'booking_cost',
        'photos',
        'operational_manual',
        'service_manual',
        'video_link',
        'public_visibility_status',
    ];

    protected $casts = [
        'booking_cost' => 'array'
    ];


    public function instrumentCategory()
    {
        return $this->belongsTo(InstrumentCategory::class);
    }

    public function lab()
    {
        return $this->belongsTo(Lab::class);
    }

    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function instrumentAccessories()
    {
        return $this->hasMany(InstrumentAccessorie::class);
    }

    public function purchaseInformation()
    {
        return $this->hasOne(InstrumentPurchaseInfo::class);
    }

    public function services()
    {
        return $this->hasMany(InstrumentService::class);
    }


    public function instrumentTerm()
    {
        return $this->hasOne(InstrumentTerm::class);
    }

    public function getNextBookingDateTime()
    {
        $today = now()->toDateString();
        $currentTime = now()->format('H:i:s');

        $nextBooking = $this->bookings()
            ->where('status', 'confirmed')
            ->whereHas('slots') // Ensure booking has at least one slot
            ->with(['slots' => function($query) {
                $query->orderBy('start_time', 'asc');
            }])
            ->where(function($query) use ($today, $currentTime) {
                // Future dates
                $query->where('date', '>', $today)
                    // Or today but future time
                    ->orWhere(function($q) use ($today, $currentTime) {
                        $q->where('date', $today)
                            ->whereHas('slots', function($slotQuery) use ($currentTime) {
                                $slotQuery->where('start_time', '>', $currentTime);
                            });
                    });
            })
            ->orderBy('date', 'asc')
            ->first();

        if ($nextBooking && $nextBooking->slots->isNotEmpty()) {
            // Get the first slot (they're ordered by start_time)
            $firstSlot = $nextBooking->slots->first();
            $bookingDate = \Carbon\Carbon::parse($nextBooking->date)->format('d M Y');
            $startTime = \Carbon\Carbon::parse($firstSlot->start_time)->format('h:i A');

            return $bookingDate . ', ' . $startTime;
        }

        return 'No upcoming bookings';
    }

    public function getNextAvailableSlot()
    {
        $today = now()->toDateString();
        $currentTime = now()->format('H:i:s');

        // First try to find a slot today
        $nextSlot = $this->findAvailableSlot($today, $currentTime);
        if ($nextSlot) {
            $formattedDate = now()->format('d M Y');
            $startTime = \Carbon\Carbon::parse($nextSlot->start_time)->format('h:i A');
            return "$formattedDate, $startTime";
        }

        // If nothing today, try tomorrow
        $tomorrow = now()->addDay()->toDateString();
        $nextSlot = $this->findAvailableSlot($tomorrow);
        if ($nextSlot) {
            $formattedDate = now()->addDay()->format('d M Y');
            $startTime = \Carbon\Carbon::parse($nextSlot->start_time)->format('h:i A');
            return "$formattedDate, $startTime";
        }

        return 'No available slots';
    }

    public function findAvailableSlot($date, $minTime = null)
    {
        return Slot::orderBy('start_time')
            ->whereDoesntHave('bookings', function($query) use ($date) {
                $query->where('bookings.instrument_id', $this->id)
                    ->where('bookings.date', $date)
                    ->where('bookings.status', 'confirmed');
            })
            ->when($minTime && $date === now()->toDateString(), function($query) use ($minTime) {
                $query->where('start_time', '>', $minTime);
            })
            ->first();
    }

    public function anyPendingService()
    {
        return $this->services()->where('status', 'pending')->exists();
    }

    public function technician()
    {
        return $this->belongsTo(Technician::class);
    }
}
