<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Helpers\UnitConverter;

class BatchItem extends Model
{
    protected $fillable = [
        'stock_batch_id',
        'inventory_item_id',
        'item_variant_id',
        'initial_quantity',
        'current_quantity',
        'unit_cost',
        'notes',
        'expiry_date',
        'manufacture_date'
    ];

    protected $casts = [
        'initial_quantity' => 'decimal:4',
        'current_quantity' => 'decimal:4',
        'unit_cost' => 'decimal:2',
        'expiry_date' => 'date',
        'manufacture_date' => 'date'
    ];

    public function stockBatch()
    {
        return $this->belongsTo(StockBatch::class);
    }

    public function inventoryItem()
    {
        return $this->belongsTo(InventoryItem::class);
    }

    public function variant()
    {
        return $this->belongsTo(ItemVariant::class, 'item_variant_id');
    }

    public function transactions()
    {
        return $this->hasMany(StockTransaction::class, 'batch_item_id');
    }

    public function getUsagePercentageAttribute()
    {
        if ($this->initial_quantity == 0) {
            return 0;
        }
        return (($this->initial_quantity - $this->current_quantity) / $this->initial_quantity) * 100;
    }

    public function getUsedQuantityAttribute()
    {
        return $this->initial_quantity - $this->current_quantity;
    }

    public function isOutOfStock()
    {
        return $this->current_quantity <= 0;
    }

    public function isLowStock($threshold = 0.2)
    {
        return $this->current_quantity <= ($this->initial_quantity * $threshold);
    }

    public function isExpired()
    {
        return $this->expiry_date && $this->expiry_date->isPast();
    }

    public function isExpiringSoon($days = 30)
    {
        if (!$this->expiry_date) {
            return false;
        }
        
        return $this->expiry_date->isFuture() && 
               $this->expiry_date->diffInDays(now()) <= $days;
    }

    public function getFormattedCurrentQuantityAttribute()
    {
        if ($this->variant) {
            return UnitConverter::formatBestUnit(
                $this->current_quantity, 
                UnitConverter::getBaseUnit($this->variant->unit_type)
            );
        }
        return number_format($this->current_quantity, 2);
    }

    public function getFormattedInitialQuantityAttribute()
    {
        if ($this->variant) {
            return UnitConverter::formatBestUnit(
                $this->initial_quantity, 
                UnitConverter::getBaseUnit($this->variant->unit_type)
            );
        }
        return number_format($this->initial_quantity, 2);
    }

    public function getFormattedUsedQuantityAttribute()
    {
        if ($this->variant) {
            return UnitConverter::formatBestUnit(
                $this->used_quantity, 
                UnitConverter::getBaseUnit($this->variant->unit_type)
            );
        }
        return number_format($this->used_quantity, 2);
    }
}
