<?php

namespace App\Livewire\Usage;

use App\Models\Booking;
use App\Models\Student;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\WithPagination;

class UserUsage extends Component
{
    public $userType;
    public $userData = [];
    public $studentUsageData = [];
    public $perPage = 10;
    public $instrumentPage = 1;
    public $studentPage = 1;

    public function mount()
    {
        $this->determineUserType();
        $this->loadUserData();
    }

    private function determineUserType()
    {
        $user = auth()->user();
        
        if ($user->hasRole('student')) {
            $this->userType = 'student';
        } elseif ($user->hasRole('pi')) {
            $this->userType = 'pi';
        } else {
            $this->userType = 'other';
        }
    }

    private function loadUserData()
    {
        if ($this->userType === 'student') {
            $this->loadStudentData();
        } elseif ($this->userType === 'pi') {
            $this->loadPiData();
        }
    }

    private function loadStudentData()
    {
        $student = auth()->user()->students->first();
        
        if (!$student) {
            return;
        }

        $bookings = Booking::where('student_id', $student->id)
            ->where('status', 'confirmed')
            ->with(['instrument', 'slots'])
            ->get();

        $instrumentUsage = $bookings->groupBy('instrument_id')->map(function ($bookings, $instrumentId) {
            $instrument = $bookings->first()->instrument;
            $totalHours = $bookings->sum(function ($booking) {
                return $booking->slots->sum(function ($slot) {
                    $start = Carbon::parse($slot->start_time);
                    $end = Carbon::parse($slot->end_time);
                    return $start->diffInHours($end);
                });
            });

            return [
                'instrument_name' => $instrument->name,
                'lab_name' => $instrument->lab->lab_name ?? 'N/A',
                'total_hours' => round($totalHours, 2),
                'booking_count' => $bookings->count(),
                'last_booking' => $bookings->sortByDesc('date')->first() ? $bookings->sortByDesc('date')->first()->date : 'No bookings'
            ];
        })->sortByDesc('total_hours');

        $this->userData = [
            'student_name' => $student->first_name . ' ' . $student->last_name,
            'academic_id' => $student->academic_id,
            'total_bookings' => $bookings->count(),
            'total_hours' => round($bookings->sum(function ($booking) {
                return $booking->slots->sum(function ($slot) {
                    $start = Carbon::parse($slot->start_time);
                    $end = Carbon::parse($slot->end_time);
                    return $start->diffInHours($end);
                });
            }), 2),
            'instruments' => $instrumentUsage
        ];
    }

    private function loadPiData()
    {
        $pi = auth()->user()->principalInvestigators->first();
        
        if (!$pi) {
            return;
        }

        $students = $pi->students;
        
        $studentData = $students->map(function ($student) {
            $bookings = Booking::where('student_id', $student->id)
                ->where('status', 'confirmed')
                ->with('slots')
                ->get();

            $totalHours = $bookings->sum(function ($booking) {
                return $booking->slots->sum(function ($slot) {
                    $start = Carbon::parse($slot->start_time);
                    $end = Carbon::parse($slot->end_time);
                    return $start->diffInHours($end);
                });
            });

            return [
                'id' => $student->id,
                'name' => $student->first_name . ' ' . $student->last_name,
                'academic_id' => $student->academic_id,
                'total_bookings' => $bookings->count(),
                'total_hours' => round($totalHours, 2),
                'last_booking' => $bookings->sortByDesc('date')->first() ? $bookings->sortByDesc('date')->first()->date : 'No bookings'
            ];
        })->sortByDesc('total_hours');

        $this->studentUsageData = $studentData->values()->all();

        $totalPiHours = collect($this->studentUsageData)->sum('total_hours');
        $totalPiBookings = collect($this->studentUsageData)->sum('total_bookings');

        $this->userData = [
            'pi_name' => $pi->getFullNameAttribute(),
            'department' => $pi->department,
            'total_students' => $students->count(),
            'total_bookings' => $totalPiBookings,
            'total_hours' => $totalPiHours
        ];
    }

    public function nextInstrumentPage()
    {
        if ($this->userType !== 'student' || !isset($this->userData['instruments'])) {
            return;
        }
        $totalPages = ceil(count($this->userData['instruments']) / $this->perPage);
        if ($this->instrumentPage < $totalPages) {
            $this->instrumentPage++;
        }
    }

    public function previousInstrumentPage()
    {
        if ($this->instrumentPage > 1) {
            $this->instrumentPage--;
        }
    }

    public function gotoInstrumentPage($page)
    {
        if ($this->userType !== 'student' || !isset($this->userData['instruments'])) {
            return;
        }
        $totalPages = ceil(count($this->userData['instruments']) / $this->perPage);
        if ($page >= 1 && $page <= $totalPages) {
            $this->instrumentPage = $page;
        }
    }

    public function nextStudentPage()
    {
        if ($this->userType !== 'pi') {
            return;
        }
        $totalPages = ceil(count($this->studentUsageData) / $this->perPage);
        if ($this->studentPage < $totalPages) {
            $this->studentPage++;
        }
    }

    public function previousStudentPage()
    {
        if ($this->studentPage > 1) {
            $this->studentPage--;
        }
    }

    public function gotoStudentPage($page)
    {
        if ($this->userType !== 'pi') {
            return;
        }
        $totalPages = ceil(count($this->studentUsageData) / $this->perPage);
        if ($page >= 1 && $page <= $totalPages) {
            $this->studentPage = $page;
        }
    }

    public function getPaginatedInstrumentDataProperty()
    {
        if ($this->userType !== 'student' || !isset($this->userData['instruments'])) {
            return collect([]);
        }
        
        $collection = collect($this->userData['instruments']);
        return $collection->forPage($this->instrumentPage, $this->perPage);
    }

    public function getPaginatedStudentDataProperty()
    {
        if ($this->userType !== 'pi') {
            return collect([]);
        }
        
        $collection = collect($this->studentUsageData);
        return $collection->forPage($this->studentPage, $this->perPage);
    }

    public function getInstrumentTotalPagesProperty()
    {
        if ($this->userType !== 'student' || !isset($this->userData['instruments'])) {
            return 0;
        }
        return ceil(count($this->userData['instruments']) / $this->perPage);
    }

    public function getStudentTotalPagesProperty()
    {
        if ($this->userType !== 'pi') {
            return 0;
        }
        return ceil(count($this->studentUsageData) / $this->perPage);
    }

    public function render()
    {
        return view('livewire.usage.user-usage');
    }
}