<?php

namespace App\Livewire\Usage;

use App\Models\Booking;
use App\Models\Instrument;
use App\Models\Student;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;

class SuperAdminUsage extends Component
{
    public $selectedDate;
    public $instrumentUsageData = [];
    public $studentUsageData = [];
    public $statsData = [];
    public $perPage = 10;
    public $instrumentPage = 1;
    public $studentPage = 1;

    public function mount()
    {
        $this->selectedDate = now()->toDateString();
        $this->loadData();
    }

    public function updatedSelectedDate()
    {
        $this->instrumentPage = 1;
        $this->studentPage = 1;
        $this->loadData();
    }

    public function loadData()
    {
        $this->loadStats();
        $this->loadInstrumentUsage();
        $this->loadStudentUsage();
    }

    private function loadStats()
    {
        $today = now()->toDateString();
        $thisMonth = now()->format('Y-m');

        // Top booked instrument
        $topInstrument = Booking::select('instrument_id', DB::raw('COUNT(*) as booking_count'))
            ->where('status', 'confirmed')
            ->groupBy('instrument_id')
            ->orderBy('booking_count', 'desc')
            ->with('instrument')
            ->first();

        // Today's bookings
        $todayBookings = Booking::where('date', $today)
            ->where('status', 'confirmed')
            ->count();

        // This month's bookings
        $thisMonthBookings = Booking::whereYear('date', now()->year)
            ->whereMonth('date', now()->month)
            ->where('status', 'confirmed')
            ->count();

        // Total usage hours today
        $todayUsageHours = Booking::where('date', $today)
            ->where('status', 'confirmed')
            ->with('slots')
            ->get()
            ->sum(function ($booking) {
                return $booking->slots->sum(function ($slot) {
                    $start = Carbon::parse($slot->start_time);
                    $end = Carbon::parse($slot->end_time);
                    return $start->diffInHours($end);
                });
            });

        $this->statsData = [
            'top_instrument' => $topInstrument ? $topInstrument->instrument->name : 'N/A',
            'top_instrument_count' => $topInstrument ? $topInstrument->booking_count : 0,
            'today_bookings' => $todayBookings,
            'month_bookings' => $thisMonthBookings,
            'today_usage_hours' => round($todayUsageHours, 2)
        ];
    }

    private function loadInstrumentUsage()
    {
        $selectedDate = $this->selectedDate;
        
        $instruments = Instrument::with(['bookings' => function ($query) use ($selectedDate) {
            $query->where('status', 'confirmed')
                ->where('date', $selectedDate)
                ->with('slots');
        }])->get();

        $instrumentData = $instruments->map(function ($instrument) {
            $todayUsage = $instrument->bookings->sum(function ($booking) {
                return $booking->slots->sum(function ($slot) {
                    $start = Carbon::parse($slot->start_time);
                    $end = Carbon::parse($slot->end_time);
                    return $start->diffInHours($end);
                });
            });

            $overallUsage = Booking::where('instrument_id', $instrument->id)
                ->where('status', 'confirmed')
                ->with('slots')
                ->get()
                ->sum(function ($booking) {
                    return $booking->slots->sum(function ($slot) {
                        $start = Carbon::parse($slot->start_time);
                        $end = Carbon::parse($slot->end_time);
                        return $start->diffInHours($end);
                    });
                });

            return [
                'id' => $instrument->id,
                'name' => $instrument->name,
                'lab' => $instrument->lab->lab_name ?? 'N/A',
                'today_usage' => round($todayUsage, 2),
                'overall_usage' => round($overallUsage, 2),
                'today_bookings' => $instrument->bookings->count()
            ];
        })->sortByDesc('today_usage');

        $this->instrumentUsageData = $instrumentData->values()->all();
    }

    private function loadStudentUsage()
    {
        $selectedDate = $this->selectedDate;
        
        $students = Student::with(['bookings' => function ($query) use ($selectedDate) {
            $query->where('status', 'confirmed')
                ->where('date', $selectedDate)
                ->with('slots');
        }, 'principalInvestigator'])->get();

        $studentData = $students->map(function ($student) {
            $todayUsage = $student->bookings->sum(function ($booking) {
                return $booking->slots->sum(function ($slot) {
                    $start = Carbon::parse($slot->start_time);
                    $end = Carbon::parse($slot->end_time);
                    return $start->diffInHours($end);
                });
            });

            $overallUsage = Booking::where('student_id', $student->id)
                ->where('status', 'confirmed')
                ->with('slots')
                ->get()
                ->sum(function ($booking) {
                    return $booking->slots->sum(function ($slot) {
                        $start = Carbon::parse($slot->start_time);
                        $end = Carbon::parse($slot->end_time);
                        return $start->diffInHours($end);
                    });
                });

            return [
                'id' => $student->id,
                'name' => $student->first_name . ' ' . $student->last_name,
                'academic_id' => $student->academic_id,
                'pi_name' => $student->principalInvestigator ? $student->principalInvestigator->getFullNameAttribute() : 'N/A',
                'today_usage' => round($todayUsage, 2),
                'overall_usage' => round($overallUsage, 2),
                'today_bookings' => $student->bookings->count()
            ];
        })->sortByDesc('today_usage');

        $this->studentUsageData = $studentData->values()->all();
    }

    public function nextInstrumentPage()
    {
        $totalPages = ceil(count($this->instrumentUsageData) / $this->perPage);
        if ($this->instrumentPage < $totalPages) {
            $this->instrumentPage++;
        }
    }

    public function previousInstrumentPage()
    {
        if ($this->instrumentPage > 1) {
            $this->instrumentPage--;
        }
    }

    public function gotoInstrumentPage($page)
    {
        $totalPages = ceil(count($this->instrumentUsageData) / $this->perPage);
        if ($page >= 1 && $page <= $totalPages) {
            $this->instrumentPage = $page;
        }
    }

    public function nextStudentPage()
    {
        $totalPages = ceil(count($this->studentUsageData) / $this->perPage);
        if ($this->studentPage < $totalPages) {
            $this->studentPage++;
        }
    }

    public function previousStudentPage()
    {
        if ($this->studentPage > 1) {
            $this->studentPage--;
        }
    }

    public function gotoStudentPage($page)
    {
        $totalPages = ceil(count($this->studentUsageData) / $this->perPage);
        if ($page >= 1 && $page <= $totalPages) {
            $this->studentPage = $page;
        }
    }

    public function getPaginatedInstrumentDataProperty()
    {
        $collection = collect($this->instrumentUsageData);
        return $collection->forPage($this->instrumentPage, $this->perPage);
    }

    public function getPaginatedStudentDataProperty()
    {
        $collection = collect($this->studentUsageData);
        return $collection->forPage($this->studentPage, $this->perPage);
    }

    public function getInstrumentTotalPagesProperty()
    {
        return ceil(count($this->instrumentUsageData) / $this->perPage);
    }

    public function getStudentTotalPagesProperty()
    {
        return ceil(count($this->studentUsageData) / $this->perPage);
    }

    public function render()
    {
        return view('livewire.usage.super-admin-usage');
    }
}