<?php

namespace App\Livewire\Technician;

use App\Mail\WelcomePrincipalInvestigatorMail;
use App\Mail\WelcomeTechnicianMail;
use App\Models\Technician;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TechnicianExport;
use Illuminate\Support\Facades\Mail;

class TechnicianList extends Component
{
    use WithPagination;

    public $showForm = false;
    public $search = '';
    public $status = 'All';
    public $isEditing = false;
    public $technicianId = null;
    public $viewTechnicianDetailView = false;
    public $viewTechnicianDetails;

    protected $listeners = [
        'technicianCreated' => 'handleTechnicianCreated',
        'technicianUpdated' => 'handleTechnicianUpdated',
        'hideViewTechnician' => 'handleHideViewTechnician',
    ];

    public function hideForm()
    {
        $this->showForm = !$this->showForm;

        if (!$this->showForm) {
            $this->isEditing = false;
            $this->technicianId = null;
            $this->dispatch('resetForm');
        }
    }

    public function editTechnician($id)
    {
        $this->technicianId = $id;
        $this->isEditing = true;
        $this->showForm = true;
        $this->dispatch('editTechnician', technicianId: $id);
    }

    public function deleteTechnician($id)
    {
        $technician = Technician::findOrFail($id);
        $technician->user->delete();
        $technician->delete();
        $this->resetPage();

        session()->flash('success', 'Technician successfully deleted.');
    }

    public function viewTechnician($id)
    {
        $this->viewTechnicianDetails = Technician::findOrFail($id);

        if ($this->viewTechnicianDetails) {
            $this->viewTechnicianDetailView = true;
        }
    }

    public function handleHideViewTechnician()
    {
        $this->viewTechnicianDetailView = false;
    }

    public function toggleStatus($id)
    {
        $technician = Technician::findOrFail($id);
        $technician->status = !$technician->status;
        $technician->save();

        session()->flash('success', 'Status successfully updated.');
    }

    public function handleTechnicianCreated()
    {
        $this->showForm = false;
        session()->flash('success', 'Technician successfully created.');
    }

    public function handleTechnicianUpdated()
    {
        $this->showForm = false;
        $this->isEditing = false;
        $this->technicianId = null;
        session()->flash('success', 'Technician successfully updated.');
    }

    public function resendWelcomeEmail($id)
    {
        $technician = Technician::findOrFail($id);
        $user = $technician->user;

        $newPassword = Str::random(10);
        $user->password = Hash::make($newPassword);
        $user->save();

        Mail::to($user->email)->queue(new WelcomePrincipalInvestigatorMail($user, $newPassword));

        session()->flash('success', 'Welcome email resent successfully with new password.');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStatus()
    {
        $this->resetPage();
    }

    public function exportToExcel()
    {
        $query = $this->getFilteredQuery();
        return Excel::download(new TechnicianExport($query), 'technicians.xlsx');
    }

    public function exportToPdf()
    {
        $query = $this->getFilteredQuery();
        $technicianList = $query->get();

        $pdf = PDF::loadView('exports.technician-pdf', [
            'technicianList' => $technicianList
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'technicians.pdf');
    }

    private function getFilteredQuery()
    {
        $query = Technician::query()->latest();
        return $query;
    }

    public function render()
    {
        $query = Technician::query()->latest();

        if ($this->search) {
            $query->where(function($q) {
                $q->where('first_name', 'like', '%' . $this->search . '%')
                    ->orWhere('last_name', 'like', '%' . $this->search . '%')
                    ->orWhere('title', 'like', '%' . $this->search . '%')
                    ->orWhere('email', 'like', '%' . $this->search . '%')
                    ->orWhere('phone', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->status !== 'All') {
            $query->where('status', $this->status);
        }

        $technicianList = $query->paginate(10);

        return view('livewire.technician.technician-list', [
            'technicianList' => $technicianList,
            'isEditing' => $this->isEditing,
            'viewTechnicianDetails' => $this->viewTechnicianDetails,
        ]);
    }
}
