<?php

namespace App\Livewire\Technician;

use App\Mail\WelcomePrincipalInvestigatorMail;
use App\Mail\WelcomeTechnicianMail;
use App\Models\User;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use App\Models\Technician;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;

class TechnicianForm extends Component
{
    use WithFileUploads;

    // Form Fields
    public $title;
    public $first_name;
    public $last_name;
    public $department;
    public $designation;
    public $email;
    public $alt_email;
    public $phone;
    public $mobile;
    public $office_address;
    public $specialization;
    public $qualification;
    public $profile_photo;
    public $status = 1;

    public $isEditing = false;
    public $technicianId = null;
    public $existingPhoto = null;

    protected $listeners = [
        'editTechnician' => 'loadTechnicianData',
        'resetForm' => 'resetForm'
    ];

    public function updated($propertyName)
    {
        if ($propertyName === 'email') {
            $this->email = strtolower($this->email);
        }

        $this->validateOnly($propertyName);
    }

    protected function rules()
    {
        $rules = [
            'title' => 'nullable',
            'first_name' => 'required|string|max:100',
            'last_name' => 'nullable|string|max:100',
            'department' => 'nullable|string',
            'designation' => 'nullable|string',
            'alt_email' => 'nullable|email|max:100',
            'phone' => 'nullable|string|min:10|max:15|regex:/^\d{1,15}$/',
            'mobile' => 'nullable|string|min:10|max:15|regex:/^\d{1,15}$/',
            'specialization' => 'nullable|string|max:255',
            'qualification' => 'nullable|string',
            'office_address' => 'nullable|string',
        ];

        if (!$this->isEditing) {
            $rules['profile_photo'] = 'nullable|image|max:1024';
            $rules['email'] = 'required|email|max:100|unique:users,email';
        } elseif ($this->profile_photo) {
            $rules['profile_photo'] = 'nullable|image|max:1024';
        }

        return $rules;
    }

    protected function messages()
    {
        return [
            'email.unique' => 'This email is already registered in our system.',
            'alt_email.different' => 'Alternative email must be different from primary email.',
            'profile_photo.max' => 'The profile photo must not exceed 1MB.',
        ];
    }

    public function loadTechnicianData($technicianId)
    {
        $this->isEditing = true;
        $this->technicianId = $technicianId;

        $technician = Technician::findOrFail($technicianId);

        $this->title = $technician->title;
        $this->first_name = $technician->first_name;
        $this->last_name = $technician->last_name;
        $this->department = $technician->department;
        $this->designation = $technician->designation;
        $this->email = $technician->email;
        $this->alt_email = $technician->alt_email;
        $this->phone = $technician->phone;
        $this->mobile = $technician->mobile;
        $this->specialization = $technician->specialization;
        $this->qualification = $technician->qualification;
        $this->office_address = $technician->office_address;
        $this->existingPhoto = $technician->profile_photo;
    }

    public function resetForm()
    {
        $this->reset([
            'profile_photo', 'title', 'first_name', 'last_name', 'department',
            'designation', 'email', 'alt_email', 'phone', 'mobile',
            'specialization', 'qualification', 'office_address',
            'isEditing', 'technicianId', 'existingPhoto'
        ]);

        $this->resetValidation();
    }

    public function submit()
    {
        $this->validate();

        if ($this->isEditing) {
            $this->updateTechnician();
        } else {
            $this->createTechnician();
        }
    }

    public function createTechnician()
    {
        $this->validate();

        $password = Str::random(10);

        $user = User::create([
            'name' => $this->first_name . ' ' . $this->last_name,
            'email' => $this->email,
            'password' => Hash::make($password),
        ]);

        if ($user) {
            $user->assignRole('technician');
        }

        $technicianData = [
            'user_id' => $user->id,
            'title' => $this->title,
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'department' => $this->department,
            'designation' => $this->designation,
            'email' => $this->email,
            'alt_email' => $this->alt_email,
            'phone' => $this->phone,
            'mobile' => $this->mobile,
            'office_address' => $this->office_address,
            'specialization' => $this->specialization,
            'qualification' => $this->qualification,
            'status' => $this->status,
        ];

        if ($this->profile_photo) {
            $fileName = time() . '_' . $this->profile_photo->getClientOriginalName();
            $filePath = $this->profile_photo->storeAs('profile_photos', $fileName, 'public');
            $technicianData['profile_photo'] = $filePath;
        }

        Technician::create($technicianData);

        Mail::to($user->email)->queue(new WelcomePrincipalInvestigatorMail($user, $password));

        $this->resetForm();

        $this->dispatch('technicianCreated');
    }

    private function updateTechnician()
    {
        $technician = Technician::findOrFail($this->technicianId);

        $photoPath = $this->existingPhoto;

        if ($this->profile_photo) {
            if ($technician->profile_photo && Storage::disk('public')->exists($technician->profile_photo)) {
                Storage::disk('public')->delete($technician->profile_photo);
            }

            $photoPath = $this->profile_photo->store('profile_photos', 'public');
        }

        $technician->user->update([
            'name' => $this->first_name . ' ' . $this->last_name,
        ]);

        $technician->update([
            'profile_photo' => $photoPath,
            'title' => $this->title,
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'department' => $this->department,
            'designation' => $this->designation,
            'alt_email' => $this->alt_email,
            'phone' => $this->phone,
            'mobile' => $this->mobile,
            'specialization' => $this->specialization,
            'qualification' => $this->qualification,
            'office_address' => $this->office_address,
        ]);

        $this->resetForm();
        session()->flash('success', 'Technician updated successfully!');

        $this->dispatch('technicianUpdated');
    }

    public function render()
    {
        return view('livewire.technician.technician-form');
    }
}
