<?php

namespace App\Livewire;

use App\Models\Booking;
use Carbon\Carbon;
use Livewire\Component;

class SuperAdminDashboard extends Component
{
    public $totalLabs = 0;
    public $totalPis = 0;
    public $totalStudents = 0;
    public $totalInstruments = 0;
    public $instrumentUnderMaintenance = 0;
    public $instrumentCategories = 0;
    public $activeBookings = 0;
    public $underService = 0;
    public $totalComplaints = 0;
    public $thisMonthCollection = 0;
    public $calendarData = [];
    public $currentMonthName;
    public $topBookedInstruments = [];
    public $mostServicedInstruments = [];
    public $todaysBookings = [];

    public function mount()
    {
        $this->generateCalendarData();
    }

    public function render()
    {
        $this->loadStatistics();
        return view('livewire.super-admin-dashboard');
    }

    protected function loadStatistics()
    {
        $this->totalLabs = \App\Models\Lab::count();
        $this->totalPis = \App\Models\PrincipalInvestigator::count();
        $this->totalStudents = \App\Models\Student::count();
        $this->totalInstruments = \App\Models\Instrument::count();
        $this->instrumentUnderMaintenance = \App\Models\Instrument::where('operating_status', 'under_maintenance')->count();
        $this->instrumentCategories = \App\Models\InstrumentCategory::count();
        $this->underService = \App\Models\InstrumentService::where('status', 'pending')->count();
        $this->totalComplaints = \App\Models\InstrumentComplaint::where('status', 'pending')->count();

        $this->loadTopBookedInstruments();
        $this->loadActiveBookings();
        $this->loadThisMonthCollection();
        $this->loadMostServicedInstruments();
        $this->loadTodaysBookings();
    }

    protected function generateCalendarData()
    {
        $now = Carbon::now();
        $this->currentMonthName = $now->format('F Y');
        $year = $now->year;
        $month = $now->month;

        $daysInMonth = $now->daysInMonth;
        $firstDay = $now->copy()->startOfMonth()->dayOfWeek;

        $calendar = [];
        $dayCount = 1;

        for ($i = 0; $i < 6; $i++) {
            $week = [];

            for ($j = 0; $j < 7; $j++) {
                if (($i === 0 && $j < $firstDay) || $dayCount > $daysInMonth) {
                    $week[] = null;
                } else {
                    $date = Carbon::create($year, $month, $dayCount)->format('Y-m-d');
                    $week[] = [
                        'day' => $dayCount,
                        'date' => $date,
                        'confirmed' => $this->getBookingCount($date, 'confirmed'),
                        'cancelled' => $this->getBookingCount($date, 'cancelled')
                    ];
                    $dayCount++;
                }
            }

            $calendar[] = $week;

            if ($dayCount > $daysInMonth) {
                break;
            }
        }

        $this->calendarData = $calendar;
    }

    protected function getBookingCount($date, $type)
    {
        return Booking::whereDate('date', $date)
            ->where('status', $type)
            ->count();
    }

    protected function loadTopBookedInstruments()
    {
        $this->topBookedInstruments = Booking::where('status', 'confirmed')
            ->select('instrument_id')
            ->selectRaw('COUNT(*) as booking_count')
            ->with('instrument:id,name')
            ->groupBy('instrument_id')
            ->orderByDesc('booking_count')
            ->limit(3)
            ->get()
            ->map(function($booking) {
                return [
                    'name' => $booking->instrument->name,
                    'count' => $booking->booking_count
                ];
            })
            ->toArray();
    }

    protected function loadActiveBookings()
    {
        $currentDate = Carbon::now()->toDateString();
        $currentTime = Carbon::now()->toTimeString();

        $this->activeBookings = Booking::where('status', 'confirmed')
            ->where(function($query) use ($currentDate, $currentTime) {
                $query->where('date', '>', $currentDate)
                    ->orWhere(function($q) use ($currentDate, $currentTime) {
                        $q->where('date', $currentDate)
                            ->whereHas('slots', function($sq) use ($currentTime) {
                                $sq->where('end_time', '>', $currentTime);
                            }, '>=', 1);
                    });
            })
            ->count();
    }

    protected function loadThisMonthCollection()
    {
        $startOfMonth = Carbon::now()->startOfMonth()->toDateString();
        $endOfMonth = Carbon::now()->endOfMonth()->toDateString();

        $totalCollection = Booking::where('status', 'confirmed')
            ->whereBetween('date', [$startOfMonth, $endOfMonth])
            ->sum('booking_cost');

        $this->thisMonthCollection = round($totalCollection, 2);
    }

    protected function loadMostServicedInstruments()
    {
        $this->mostServicedInstruments = \App\Models\InstrumentService::select('instrument_id')
            ->selectRaw('COUNT(*) as service_count')
            ->with('instrument:id,name')
            ->where('status', 'completed')
            ->groupBy('instrument_id')
            ->orderByDesc('service_count')
            ->limit(3)
            ->get()
            ->map(function($service) {
                return [
                    'name' => $service->instrument ? $service->instrument->name : 'Unknown Instrument #' . $service->instrument_id,
                    'count' => (int)$service->service_count
                ];
            })
            ->toArray();
    }

    protected function loadTodaysBookings()
    {
        $today = Carbon::now()->toDateString();
        
        $this->todaysBookings = Booking::where('status', 'confirmed')
            ->whereDate('date', $today)
            ->with(['instrument:id,name', 'student:id,first_name,last_name', 'slots'])
            ->get()
            ->map(function($booking) {
                $slots = $booking->slots->sortBy('start_time');
                $firstSlot = $slots->first();
                $lastSlot = $slots->last();
                
                return [
                    'id' => $booking->id,
                    'instrument_name' => $booking->instrument->name,
                    'student_name' => $booking->student->first_name . ' ' . $booking->student->last_name,
                    'start_time' => $firstSlot ? $firstSlot->start_time : null,
                    'end_time' => $lastSlot ? $lastSlot->end_time : null,
                    'booking_date' => $booking->date
                ];
            })
            ->sortBy('start_time')
            ->values()
            ->toArray();
    }
}
