<?php

namespace App\Livewire\RolesPermissions;

use Livewire\Component;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RolesList extends Component
{
    public $addRoleDiv = false;
    public $showAssignPermissionsDiv = false;
    public $name = '';
    public $roleId = null;
    public $isEditing = false;
    public $selectedPermissions = [];
    public $permissions = [];
    public $showAssignRoleToUserDiv = false;
    public $selectedUserId = '';
    public $users = [];
    public $selectedRoleId = null;

    protected $rules = [
        'name' => 'required|string|max:255',
        'selectedPermissions' => 'array',
        'selectedUserId' => 'required|exists:users,id'
    ];

    public function mount()
    {
        $this->permissions = Permission::all();
        $this->users = \App\Models\User::all();
    }

    public function showAddRoleDivFunction()
    {
        $this->addRoleDiv = true;
    }

    public function editRole($id)
    {
        $role = Role::findOrFail($id);
        $this->roleId = $role->id;
        $this->name = $role->name;
        $this->isEditing = true;
        $this->addRoleDiv = true;
    }

    protected function isProtectedRole($roleName)
    {
        return in_array($roleName, ['super_admin', 'pi', 'student']);
    }

    public function deleteRole($id)
    {
        $role = Role::findOrFail($id);
        if ($this->isProtectedRole($role->name)) {
            session()->flash('error', 'Cannot delete protected role!');
            return;
        }
        $role->delete();
        session()->flash('success', 'Role deleted successfully!');
    }

    public function assignPermissions($id)
    {
        $role = Role::findOrFail($id);
        $this->roleId = $role->id;
        $this->selectedPermissions = $role->permissions->pluck('name')->toArray();
        $this->showAssignPermissionsDiv = true;
    }

    public function savePermissions()
    {
        $role = Role::findOrFail($this->roleId);
        $role->syncPermissions($this->selectedPermissions);
        session()->flash('success', 'Permissions assigned successfully!');
        $this->showAssignPermissionsDiv = false;
    }

    public function assignRoleToUser($id)
    {
        $this->selectedRoleId = $id;
        $this->showAssignRoleToUserDiv = true;
    }

    public function assignRoleToUserSubmit()
    {
        $this->validate([
            'selectedUserId' => 'required|exists:users,id'
        ]);

        try {
            $user = \App\Models\User::findOrFail($this->selectedUserId);
            $role = Role::findOrFail($this->selectedRoleId);
            
            // Get user's current roles
            $currentRoles = $user->roles;
            
            if ($currentRoles->count() > 0) {
                // User has existing roles, remove them first
                $user->roles()->detach();
                $user->assignRole($role);
                session()->flash('success', 'User role updated successfully!');
            } else {
                // User has no roles, assign new role
                $user->assignRole($role);
                session()->flash('success', 'Role assigned to user successfully!');
            }

            $this->resetForm();
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to assign role to user.');
        }
    }

    public function saveRole()
    {
        $this->validate([
            'name' => 'required'
        ]);

        if ($this->isEditing) {
            $role = Role::findOrFail($this->roleId);
            $role->update(['name' => $this->name]);
            session()->flash('success', 'Role updated successfully!');
        } else {
            Role::create(['name' => $this->name, 'guard_name' => 'web']);
            session()->flash('success', 'Role added successfully!');
        }

        $this->resetForm();
        $this->addRoleDiv = false;
    }

    public function resetForm()
    {
        $this->reset(['name', 'roleId', 'isEditing', 'selectedPermissions', 'selectedUserId', 'selectedRoleId']);
        $this->resetValidation();
        $this->addRoleDiv = false;
        $this->showAssignPermissionsDiv = false;
        $this->showAssignRoleToUserDiv = false;
    }

    public function deleteSpecificPermissionForRole($roleId, $permissionName)
    {
        try {
            $role = Role::findOrFail($roleId);
            $role->revokePermissionTo($permissionName);
            session()->flash('success', "Permission '$permissionName' removed from role successfully!");
        } catch (\Exception $e) {
            session()->flash('error', 'Failed to remove permission from role.');
        }
    }

    public function render()
    {
        $roles = Role::all();
        return view('livewire.roles-permissions.roles-list', [
            'roles' => $roles
        ]);
    }


}


