<?php

namespace App\Livewire\Inventory;

use App\Models\Supplier;
use App\Exports\SupplierExport;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class SupplierList extends Component
{
    use WithPagination;

    public $search = '';
    public $showModal = false;
    public $editMode = false;
    public $supplierId = null;
    
    // Form fields
    public $name = '';
    public $contact_person = '';
    public $email = '';
    public $phone = '';
    public $address = '';

    protected function rules()
    {
        return [
            'name' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\s\-_\.\&\(\)]+$/',
                'unique:suppliers,name,' . ($this->supplierId ?? 'NULL')
            ],
            'contact_person' => [
                'nullable',
                'string',
                'max:255',
                'regex:/^[a-zA-Z\s\-\.]+$/'
            ],
            'email' => [
                'nullable',
                'email:rfc,dns',
                'max:255',
                'unique:suppliers,email,' . ($this->supplierId ?? 'NULL')
            ],
            'phone' => [
                'nullable',
                'string',
                'max:255',
                'regex:/^[\+]?[0-9\s\-\(\)\.]+$/'
            ],
            'address' => 'nullable|string|max:500',
        ];
    }

    protected $messages = [
        'name.required' => 'Supplier name is required.',
        'name.string' => 'Supplier name must be valid text.',
        'name.max' => 'Supplier name cannot exceed 255 characters.',
        'name.regex' => 'Supplier name can only contain letters, numbers, spaces, hyphens, underscores, periods, ampersands, and parentheses.',
        'name.unique' => 'A supplier with this name already exists.',
        'contact_person.string' => 'Contact person must be valid text.',
        'contact_person.max' => 'Contact person name cannot exceed 255 characters.',
        'contact_person.regex' => 'Contact person name can only contain letters, spaces, hyphens, and periods.',
        'email.email' => 'Please enter a valid email address.',
        'email.max' => 'Email address cannot exceed 255 characters.',
        'email.unique' => 'This email address is already associated with another supplier.',
        'phone.string' => 'Phone number must be valid text.',
        'phone.max' => 'Phone number cannot exceed 255 characters.',
        'phone.regex' => 'Phone number can only contain numbers, spaces, hyphens, parentheses, periods, and plus sign.',
        'address.string' => 'Address must be valid text.',
        'address.max' => 'Address cannot exceed 500 characters.',
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->editMode = false;
        $this->showModal = true;
    }

    public function openEditModal($supplierId)
    {
        $supplier = Supplier::findOrFail($supplierId);
        
        $this->supplierId = $supplier->id;
        $this->name = $supplier->name;
        $this->contact_person = $supplier->contact_person;
        $this->email = $supplier->email;
        $this->phone = $supplier->phone;
        $this->address = $supplier->address;
        
        $this->editMode = true;
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function resetForm()
    {
        $this->supplierId = null;
        $this->name = '';
        $this->contact_person = '';
        $this->email = '';
        $this->phone = '';
        $this->address = '';
        $this->resetErrorBag();
    }



    public function delete($supplierId)
    {
        try {
            $supplier = Supplier::findOrFail($supplierId);
            
            // Enhanced dependency checking
            $dependencyCheck = $this->checkSupplierDependencies($supplier);
            if (!$dependencyCheck['canDelete']) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Cannot Delete!',
                    'message' => $dependencyCheck['message']
                ]);
                return;
            }
            
            $supplier->delete();
            
            // Toast notification for success
            $this->dispatch('toastMagic', [
                'status' => 'success',
                'title' => 'Success!',
                'message' => 'Supplier deleted successfully.'
            ]);

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'Supplier not found.'
            ]);
        } catch (\Exception $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'An error occurred while deleting the supplier. Please try again.'
            ]);
            \Log::error('Supplier deletion error: ' . $e->getMessage());
        }
    }

    /**
     * Check if supplier can be deleted by examining dependencies
     */
    private function checkSupplierDependencies(Supplier $supplier): array
    {
        $stockBatchCount = $supplier->stockBatches()->count();
        
        if ($stockBatchCount > 0) {
            // Get some batch numbers for reference
            $batchNumbers = $supplier->stockBatches()
                ->limit(3)
                ->pluck('batch_number')
                ->toArray();
            
            $message = "Cannot delete supplier '{$supplier->name}'. It has {$stockBatchCount} associated stock batch(es)";
            if ($stockBatchCount <= 3) {
                $message .= ': ' . implode(', ', $batchNumbers);
            } else {
                $message .= ' including: ' . implode(', ', $batchNumbers) . ' and ' . ($stockBatchCount - 3) . ' more';
            }
            $message .= '. Please remove or reassign these batches first.';
            
            return [
                'canDelete' => false,
                'message' => $message
            ];
        }

        return [
            'canDelete' => true,
            'message' => ''
        ];
    }

    /**
     * Real-time validation methods
     */
    public function updatedName()
    {
        $this->validateOnly('name');
    }

    public function updatedContactPerson()
    {
        $this->validateOnly('contact_person');
    }

    public function updatedEmail()
    {
        $this->validateOnly('email');
    }

    public function updatedPhone()
    {
        $this->validateOnly('phone');
    }

    public function updatedAddress()
    {
        $this->validateOnly('address');
    }

    /**
     * Enhanced save method with better error handling
     */
    public function save()
    {
        try {
            $this->validate();

            if ($this->editMode) {
                $supplier = Supplier::findOrFail($this->supplierId);
                $supplier->update([
                    'name' => trim($this->name),
                    'contact_person' => trim($this->contact_person),
                    'email' => trim($this->email),
                    'phone' => trim($this->phone),
                    'address' => trim($this->address),
                ]);
                
                // Toast notification for success
                $this->dispatch('toastMagic', [
                    'status' => 'success',
                    'title' => 'Success!',
                    'message' => 'Supplier updated successfully.'
                ]);
            } else {
                Supplier::create([
                    'name' => trim($this->name),
                    'contact_person' => trim($this->contact_person),
                    'email' => trim($this->email),
                    'phone' => trim($this->phone),
                    'address' => trim($this->address),
                ]);
                
                // Toast notification for success
                $this->dispatch('toastMagic', [
                    'status' => 'success',
                    'title' => 'Success!',
                    'message' => 'Supplier created successfully.'
                ]);
            }

            $this->closeModal();

        } catch (\Illuminate\Validation\ValidationException $e) {
            // Re-throw validation exceptions to show field errors
            throw $e;
        } catch (\Exception $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'An error occurred while saving the supplier. Please try again.'
            ]);
            \Log::error('Supplier save error: ' . $e->getMessage());
        }
    }

    public function exportSuppliers()
    {
        $query = Supplier::query()
            ->withCount('stockBatches')
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('contact_person', 'like', '%' . $this->search . '%')
                      ->orWhere('email', 'like', '%' . $this->search . '%')
                      ->orWhere('phone', 'like', '%' . $this->search . '%');
                });
            });

        return Excel::download(new SupplierExport($query), 'suppliers.xlsx');
    }

    public function exportSuppliersPdf()
    {
        $data = Supplier::query()
            ->with('stockBatches')
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('contact_person', 'like', '%' . $this->search . '%')
                      ->orWhere('email', 'like', '%' . $this->search . '%')
                      ->orWhere('phone', 'like', '%' . $this->search . '%');
                });
            })
            ->orderBy('name')
            ->get();

        $pdf = Pdf::loadView('exports.supplier-pdf', [
            'data' => $data,
            'title' => 'Suppliers Report',
            'generatedAt' => now()->format('Y-m-d H:i:s')
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'suppliers_report_' . now()->format('Y-m-d_H-i-s') . '.pdf');
    }

    public function render()
    {
        $suppliers = Supplier::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('contact_person', 'like', '%' . $this->search . '%')
                      ->orWhere('email', 'like', '%' . $this->search . '%')
                      ->orWhere('phone', 'like', '%' . $this->search . '%');
                });
            })
            ->orderBy('name')
            ->paginate(10);

        return view('livewire.inventory.supplier-list', compact('suppliers'));
    }
}