<?php

namespace App\Livewire\Inventory;

use App\Models\Boxfield;
use App\Models\Item;
use App\Models\ItemFile;
use App\Models\Lab;
use App\Models\SlotFile;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Barryvdh\DomPDF\Facade\Pdf;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;

class Storage extends Component
{
    public $box_name;
    public $lab_id;
    public $status = 1;

    public $showAddNewBoxModel = false;
    public $isEditing = false;
    public $boxId = null;
    public $availableLabs = [];
    
    // QR Code properties
    public $showQrCodeModal = false;
    public $qrCodeData;
    public $qrCodeCount = 1;
    public $selectedBox;
    
    // Search properties
    public $search = '';
    public $searchResults = [];

    protected $listeners = [
        'editBox' => 'loadBoxData',
        'resetForm' => 'resetForm'
    ];

    public function mount()
    {
        $this->loadAvailableLabs();
    }

    protected function rules()
    {
        return [
            'box_name' => 'required|string|max:255',
            'lab_id' => 'required|exists:labs,id',
        ];
    }

    public function loadAvailableLabs()
    {
        $currentUser = Auth::user();
        
        if ($currentUser->hasRole('super_admin')) {
            // Super admin can see all labs
            $this->availableLabs = Lab::with('principalInvestigator')
                                     ->where('status', 1)
                                     ->get()
                                     ->map(function ($lab) {
                                         return [
                                             'id' => $lab->id,
                                             'name' => $lab->lab_name,
                                             'pi_name' => $lab->principalInvestigator ? 
                                                         $lab->principalInvestigator->first_name . ' ' . $lab->principalInvestigator->last_name : 
                                                         'No PI Assigned',
                                             'department' => $lab->department,
                                             'building' => $lab->building,
                                             'room' => $lab->room_number
                                         ];
                                     });
        } elseif ($currentUser->hasRole('pi')) {
            // PI can see only their own labs
            $pi = $currentUser->principalInvestigators->first();
            if ($pi) {
                $this->availableLabs = Lab::where('principal_investigator_id', $pi->id)
                                         ->where('status', 1)
                                         ->get()
                                         ->map(function ($lab) {
                                             return [
                                                 'id' => $lab->id,
                                                 'name' => $lab->lab_name,
                                                 'pi_name' => $lab->principalInvestigator ? 
                                                             $lab->principalInvestigator->first_name . ' ' . $lab->principalInvestigator->last_name : 
                                                             'No PI Assigned',
                                                 'department' => $lab->department,
                                                 'building' => $lab->building,
                                                 'room' => $lab->room_number
                                             ];
                                         });
            }
        }
    }

    public function toggleShowAddNewBoxModel()
    {
        $this->showAddNewBoxModel = true;
    }

    public function closeAddNewBoxModel()
    {
        $this->showAddNewBoxModel = false;
        $this->resetForm();
    }

    public function loadBoxData($boxId)
    {
        $this->boxId = $boxId;
        $this->isEditing = true;
        $box = Boxfield::findOrFail($boxId);

        $this->box_name = $box->box_name;
        $this->lab_id = $box->lab_id;
        $this->status = $box->status;
        $this->showAddNewBoxModel = true;
    }

    public function toggleStatus($boxId)
    {
        $this->boxId = $boxId;
        $this->isEditing = true;
        $box = Boxfield::findOrFail($boxId);
        $box->status = !$box->status;
        $box->save();

        session()->flash('success', 'Box status updated successfully!');
    }

    public function saveBox()
    {
        $this->validate();

        if ($this->isEditing) {
            $this->updateBox();
        } else {
            $this->createBox();
        }
    }

    public function createBox()
    {
        $box = new Boxfield();
        $box->lab_id = $this->lab_id;
        $box->box_name = $this->box_name;
        $box->status = $this->status;
        $box->save();

        $this->resetForm();
        $this->closeAddNewBoxModel();
        session()->flash('success', 'Box created successfully!');

        $this->dispatch('boxCreated');
    }

    public function updateBox()
    {
        $box = Boxfield::findOrFail($this->boxId);
        $box->lab_id = $this->lab_id;
        $box->box_name = $this->box_name;
        $box->status = $this->status;
        $box->save();

        $this->resetForm();
        $this->closeAddNewBoxModel();
        session()->flash('success', 'Box updated successfully!');

        $this->dispatch('boxUpdated');
    }

    public function deleteBox($boxId)
    {
        $box = Boxfield::findOrFail($boxId);
        $box->delete();

        session()->flash('success', 'Box deleted successfully!');
        $this->dispatch('boxDeleted');
    }

    public function generateQrCode($boxId)
    {
        $box = Boxfield::with(['lab.principalInvestigator', 'subBoxes.itemsSlots.items', 'subBoxes.itemsSlots.slotFiles'])
                       ->findOrFail($boxId);

        // Generate URL with encoded ID
        $url = route('inventory.box-info', encode_id($boxId));

        // Generate QR code
        $rendererStyle = new RendererStyle(400);
        $imageRenderer = new ImageRenderer($rendererStyle, new SvgImageBackEnd());
        $writer = new Writer($imageRenderer);
        $svgContent = $writer->writeString($url);

        $this->qrCodeData = 'data:image/svg+xml;base64,' . base64_encode($svgContent);
        $this->selectedBox = $box;
        $this->showQrCodeModal = true;
    }

    public function printQrCodes($boxId)
    {
        $this->validate([
            'qrCodeCount' => 'required|numeric|min:1|max:100',
        ]);

        $box = Boxfield::with(['lab.principalInvestigator', 'subBoxes.itemsSlots.items', 'subBoxes.itemsSlots.slotFiles'])
                       ->findOrFail($boxId);
        $url = route('inventory.box-info', encode_id($boxId));

        // Generate QR code data for PDF
        $rendererStyle = new RendererStyle(400);
        $imageRenderer = new ImageRenderer($rendererStyle, new SvgImageBackEnd());
        $writer = new Writer($imageRenderer);
        $svgContent = $writer->writeString($url);

        $qrCodeData = 'data:image/svg+xml;base64,' . base64_encode($svgContent);

        // Calculate totals
        $totalItems = 0;
        $totalFiles = 0;
        foreach ($box->subBoxes as $subBox) {
            foreach ($subBox->itemsSlots as $itemsSlot) {
                $totalItems += $itemsSlot->items->count();
                $totalFiles += $itemsSlot->slotFiles->count();
            }
        }

        // Generate PDF
        $pdf = PDF::loadView('livewire.inventory.box-qr-code-pdf', [
            'qrCodeData' => $qrCodeData,
            'count' => $this->qrCodeCount,
            'box' => $box,
            'totalItems' => $totalItems,
            'totalFiles' => $totalFiles,
        ]);

        $filename = "box-qr-codes-{$box->box_name}-{$box->id}.pdf";
        $filename = Str::slug($filename);

        return response()->streamDownload(
            fn () => print($pdf->output()),
            $filename,
            [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="'.$filename.'"',
            ]
        );
    }

    /**
     * Search for items and files across all boxes
     */
    public function updatedSearch()
    {
        if (empty($this->search) || strlen($this->search) < 3) {
            $this->searchResults = [];
            return;
        }
        
        $currentUser = Auth::user();
        $searchTerm = '%' . $this->search . '%';
        $results = [];
        
        // Get accessible boxes based on user role
        if ($currentUser->hasRole('super_admin')) {
            $boxes = Boxfield::with(['lab', 'subBoxes.itemsSlots.user'])->get();
        } elseif ($currentUser->hasRole('pi')) {
            $pi = $currentUser->principalInvestigators->first();
            if ($pi) {
                $boxes = Boxfield::with(['lab', 'subBoxes.itemsSlots.user'])
                                ->whereHas('lab', function($query) use ($pi) {
                                    $query->where('principal_investigator_id', $pi->id);
                                })
                                ->get();
            } else {
                $boxes = collect();
            }
        } else {
            $boxes = collect();
        }
        
        // Search through items
        foreach ($boxes as $box) {
            foreach ($box->subBoxes as $subBox) {
                foreach ($subBox->itemsSlots as $itemsSlot) {
                    // Search items
                    $items = Item::where('items_slot_id', $itemsSlot->id)
                                ->where(function($query) use ($searchTerm) {
                                    $query->where('item_name', 'like', $searchTerm)
                                          ->orWhere('item_info', 'like', $searchTerm);
                                })
                                ->get();
                    
                    foreach ($items as $item) {
                        $results[] = [
                            'type' => 'item',
                            'id' => $item->id,
                            'name' => $item->item_name,
                            'info' => $item->item_info,
                            'user' => $itemsSlot->user ? $itemsSlot->user->name : 'Unassigned',
                            'path' => $box->lab->lab_name . ' → ' . $box->box_name . ' → ' . $subBox->sub_box_name . ' → ' . $itemsSlot->items_slot_name,
                            'box_id' => $box->id,
                            'sub_box_id' => $subBox->id,
                            'items_slot_id' => $itemsSlot->id,
                        ];
                    }
                    
                    // Search slot files
                    $files = SlotFile::where('items_slot_id', $itemsSlot->id)
                                    ->where('file_name', 'like', $searchTerm)
                                    ->get();
                    
                    foreach ($files as $file) {
                        $results[] = [
                            'type' => 'file',
                            'id' => $file->id,
                            'name' => $file->file_name,
                            'info' => 'File Size: ' . number_format($file->file_size / 1024, 1) . ' KB',
                            'user' => $itemsSlot->user ? $itemsSlot->user->name : 'Unassigned',
                            'path' => $box->lab->lab_name . ' → ' . $box->box_name . ' → ' . $subBox->sub_box_name . ' → ' . $itemsSlot->items_slot_name,
                            'box_id' => $box->id,
                            'sub_box_id' => $subBox->id,
                            'items_slot_id' => $itemsSlot->id,
                        ];
                    }
                }
            }
        }
        
        $this->searchResults = $results;
    }
    
    public function viewSearchResult($type, $id)
    {
        if ($type === 'item') {
            $item = Item::findOrFail($id);
            return redirect()->route('inventory.slot-items', ['id' => $item->items_slot_id]);
        } elseif ($type === 'file') {
            $file = SlotFile::findOrFail($id);
            return redirect()->route('inventory.slot-items', ['id' => $file->items_slot_id]);
        }
    }

    public function resetForm()
    {
        $this->reset([
            'box_name',
            'lab_id',
            'isEditing',
            'boxId'
        ]);

        $this->status = 1;
    }

    public function render()
    {
        $currentUser = Auth::user();
        $boxes = collect();
        
        // Only fetch boxes if not searching
        if (empty($this->search) || strlen($this->search) < 3) {
            if ($currentUser->hasRole('super_admin')) {
                // Super admin can see all boxes
                $boxes = Boxfield::with(['lab.principalInvestigator', 'subBoxes.itemsSlots.items', 'subBoxes.itemsSlots.slotFiles'])
                                 ->withCount('subBoxes')
                                 ->orderBy('created_at', 'desc')
                                 ->get();
            } elseif ($currentUser->hasRole('pi')) {
                // PI can see only boxes from their own labs
                $pi = $currentUser->principalInvestigators->first();
                if ($pi) {
                    $boxes = Boxfield::with(['lab.principalInvestigator', 'subBoxes.itemsSlots.items', 'subBoxes.itemsSlots.slotFiles'])
                                     ->withCount('subBoxes')
                                     ->whereHas('lab', function($query) use ($pi) {
                                         $query->where('principal_investigator_id', $pi->id);
                                     })
                                     ->orderBy('created_at', 'desc')
                                     ->get();
                }
            }

            // Calculate total items and files for each box
            $boxes->each(function ($box) {
                $totalItems = 0;
                $totalFiles = 0;
                
                foreach ($box->subBoxes as $subBox) {
                    foreach ($subBox->itemsSlots as $itemsSlot) {
                        $totalItems += $itemsSlot->items->count();
                        $totalFiles += $itemsSlot->slotFiles->count();
                    }
                }
                
                $box->total_items = $totalItems;
                $box->total_files = $totalFiles;
            });
        }

        return view('livewire.inventory.storage', [
            'boxes' => $boxes
        ]);
    }
}
