<?php

namespace App\Livewire\Inventory;

use App\Models\InventoryItem;
use App\Models\ItemVariant;
use App\Models\StockBatch;
use App\Models\Supplier;
use App\Models\StockTransaction;
use App\Exports\StockTransactionExport;
use App\Helpers\UnitConverter;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class StockManagement extends Component
{
    use WithPagination;

    // Stock Intake Form Properties
    public $showStockIntakeModal = false;
    public $batch_number;
    public $supplier_id;
    public $received_date;
    public $notes;
    
    // Multiple items for batch intake
    public $batchItems = [];
    public $newBatchItem = [
        'inventory_item_id' => '',
        'item_variant_id' => '',
        'initial_quantity' => '',
        'unit_cost' => 0.00,
        'notes' => '',
        'expiry_date' => '',
        'manufacture_date' => ''
    ];
    
    // Stock Out Form Properties
    public $showStockOutModal = false;
    public $selected_batch_id;
    public $used_for;
    public $conducted_by;
    public $approved_by;
    public $transaction_date;
    public $out_notes;
    public $override_expiry = false;
    
    // Multiple items for stock out
    public $stockOutItems = [];
    
    // Transaction History Properties
    public $showTransactionHistoryModal = false;
    public $selected_batch_for_history;
    
    // Batch View Properties
    public $showBatchViewModal = false;
    public $selected_batch_for_view;
    public $batchItemsForView = [];
    
    // Display properties
    public $availableVariants = [];
    
    // Search and filters
    public $search = '';
    public $itemFilter = '';
    public $supplierFilter = '';
    public $statusFilter = '';
    public $dateFrom = '';
    public $dateTo = '';
    public $transactionTypeFilter = '';
    
    protected $listeners = [
        'resetForm' => 'resetForm',
        'stockOutCompleted' => '$refresh'
    ];

    protected function rules()
    {
        $rules = [
            'batch_number' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\-_]+$/',
                'unique:stock_batches,batch_number'
            ],
            'received_date' => [
                'required',
                'date',
                'before_or_equal:today',
                'after_or_equal:' . Carbon::now()->subYears(5)->format('Y-m-d')
            ],
            'supplier_id' => 'nullable|exists:suppliers,id',
            'notes' => 'nullable|string|max:1000',
            'batchItems' => 'required|array|min:1|max:50',
            'batchItems.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'batchItems.*.initial_quantity' => [
                'required',
                'numeric',
                'min:0.001',
                'max:999999.999'
            ],
            'batchItems.*.unit_cost' => [
                'required',
                'numeric',
                'min:0',
                'max:999999.99'
            ],
            'batchItems.*.item_variant_id' => 'nullable|exists:item_variants,id',
            'batchItems.*.notes' => 'nullable|string|max:500',
            'batchItems.*.expiry_date' => [
                'nullable',
                'date',
                'after:today',
                'before:' . Carbon::now()->addYears(20)->format('Y-m-d')
            ],
            'batchItems.*.manufacture_date' => [
                'nullable',
                'date',
                'before_or_equal:today',
                'after_or_equal:' . Carbon::now()->subYears(10)->format('Y-m-d')
            ],
        ];

        // Add enhanced validation for chemical items
        foreach ($this->batchItems as $index => $batchItem) {
            if (!empty($batchItem['inventory_item_id'])) {
                $item = InventoryItem::find($batchItem['inventory_item_id']);
                if ($item && $item->isChemical()) {
                    // Chemical items require expiry and manufacture dates
                    $rules["batchItems.{$index}.expiry_date"] = [
                        'required',
                        'date',
                        'after:today',
                        'before:' . Carbon::now()->addYears(20)->format('Y-m-d')
                    ];
                    $rules["batchItems.{$index}.manufacture_date"] = [
                        'required',
                        'date',
                        'before_or_equal:today',
                        'after_or_equal:' . Carbon::now()->subYears(10)->format('Y-m-d')
                    ];
                    
                    // Validate that expiry date is after manufacture date
                    if (!empty($batchItem['manufacture_date']) && !empty($batchItem['expiry_date'])) {
                        $rules["batchItems.{$index}.expiry_date"][] = 'after:batchItems.' . $index . '.manufacture_date';
                    }
                    
                    // Chemical items with variants require variant selection
                    if ($item->variants->count() > 0) {
                        $rules["batchItems.{$index}.item_variant_id"] = 'required|exists:item_variants,id';
                        
                        // Validate that variant belongs to the selected item
                        if (!empty($batchItem['item_variant_id'])) {
                            $variant = ItemVariant::find($batchItem['item_variant_id']);
                            if ($variant && $variant->inventory_item_id != $item->id) {
                                $rules["batchItems.{$index}.item_variant_id"] .= '|in:';
                            }
                        }
                    }
                }
            }
        }

        return $rules;
    }

    protected $messages = [
        'batch_number.required' => 'Batch number is required.',
        'batch_number.string' => 'Batch number must be valid text.',
        'batch_number.max' => 'Batch number cannot exceed 255 characters.',
        'batch_number.regex' => 'Batch number can only contain letters, numbers, hyphens, and underscores.',
        'batch_number.unique' => 'This batch number already exists.',
        'received_date.required' => 'Received date is required.',
        'received_date.date' => 'Please enter a valid received date.',
        'received_date.before_or_equal' => 'Received date cannot be in the future.',
        'received_date.after_or_equal' => 'Received date cannot be more than 5 years in the past.',
        'supplier_id.exists' => 'Selected supplier is invalid.',
        'notes.string' => 'Notes must be valid text.',
        'notes.max' => 'Notes cannot exceed 1000 characters.',
        'batchItems.required' => 'At least one batch item is required.',
        'batchItems.array' => 'Batch items must be a valid list.',
        'batchItems.min' => 'At least one batch item is required.',
        'batchItems.max' => 'Cannot add more than 50 items to a single batch.',
        'batchItems.*.inventory_item_id.required' => 'Please select an item for each batch item.',
        'batchItems.*.inventory_item_id.exists' => 'Selected item is invalid.',
        'batchItems.*.initial_quantity.required' => 'Initial quantity is required for each item.',
        'batchItems.*.initial_quantity.numeric' => 'Initial quantity must be a valid number.',
        'batchItems.*.initial_quantity.min' => 'Initial quantity must be at least 0.001.',
        'batchItems.*.initial_quantity.max' => 'Initial quantity cannot exceed 999,999.999.',
        'batchItems.*.unit_cost.required' => 'Unit cost is required for each item.',
        'batchItems.*.unit_cost.numeric' => 'Unit cost must be a valid number.',
        'batchItems.*.unit_cost.min' => 'Unit cost cannot be negative.',
        'batchItems.*.unit_cost.max' => 'Unit cost cannot exceed 999,999.99.',
        'batchItems.*.item_variant_id.required' => 'Please select a variant for chemical items.',
        'batchItems.*.item_variant_id.exists' => 'Selected variant is invalid.',
        'batchItems.*.notes.string' => 'Notes must be valid text.',
        'batchItems.*.notes.max' => 'Item notes cannot exceed 500 characters.',
        'batchItems.*.expiry_date.required' => 'Expiry date is required for chemical items.',
        'batchItems.*.expiry_date.date' => 'Please enter a valid expiry date.',
        'batchItems.*.expiry_date.after' => 'Expiry date must be in the future.',
        'batchItems.*.expiry_date.before' => 'Expiry date cannot be more than 20 years in the future.',
        'batchItems.*.manufacture_date.required' => 'Manufacture date is required for chemical items.',
        'batchItems.*.manufacture_date.date' => 'Please enter a valid manufacture date.',
        'batchItems.*.manufacture_date.before_or_equal' => 'Manufacture date cannot be in the future.',
        'batchItems.*.manufacture_date.after_or_equal' => 'Manufacture date cannot be more than 10 years in the past.',
    ];

    public function mount()
    {
        $this->received_date = Carbon::today()->format('Y-m-d');
        $this->transaction_date = Carbon::today()->format('Y-m-d');
        $this->conducted_by = auth()->user()->name ?? '';
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedItemFilter()
    {
        $this->resetPage();
    }

    public function updatedSupplierFilter()
    {
        $this->resetPage();
    }

    public function updatedStatusFilter()
    {
        $this->resetPage();
    }

    public function updatedDateFrom()
    {
        $this->resetPage();
    }

    public function updatedDateTo()
    {
        $this->resetPage();
    }

    public function updatedTransactionTypeFilter()
    {
        $this->resetPage();
    }

    public function addBatchItem()
    {
        $this->batchItems[] = [
            'inventory_item_id' => '',
            'item_variant_id' => '',
            'initial_quantity' => '',
            'unit_cost' => 0.00,
            'notes' => '',
            'expiry_date' => '',
            'manufacture_date' => ''
        ];
    }

    public function removeBatchItem($index)
    {
        unset($this->batchItems[$index]);
        $this->batchItems = array_values($this->batchItems);
    }

    public function updatedBatchItems($value, $key)
    {
        // Handle item selection changes for variants
        if (strpos($key, '.inventory_item_id') !== false) {
            $index = explode('.', $key)[0];
            $itemId = $this->batchItems[$index]['inventory_item_id'];
            
            if ($itemId) {
                $item = InventoryItem::with('variants')->find($itemId);
                if ($item && $item->isChemical() && $item->variants->count() > 0) {
                    // Reset variant selection when item changes
                    $this->batchItems[$index]['item_variant_id'] = '';
                }
            }
        }
    }

    public function toggleStockIntakeModal()
    {
        $this->showStockIntakeModal = true;
        $this->generateBatchNumber();
        
        // Initialize with one empty batch item
        if (empty($this->batchItems)) {
            $this->addBatchItem();
        }
    }

    public function closeStockIntakeModal()
    {
        $this->showStockIntakeModal = false;
        $this->resetForm();
    }

    public function generateBatchNumber()
    {
        // Generate batch number format: BATCH-YYYYMMDD-XXXX
        $date = Carbon::now()->format('Ymd');
        $lastBatch = StockBatch::where('batch_number', 'like', "BATCH-{$date}-%")
            ->orderBy('batch_number', 'desc')
            ->first();
        
        if ($lastBatch) {
            $lastNumber = intval(substr($lastBatch->batch_number, -4));
            $newNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $newNumber = '0001';
        }
        
        $this->batch_number = "BATCH-{$date}-{$newNumber}";
    }

    public function validateBatchNumber()
    {
        if ($this->batch_number) {
            $exists = StockBatch::where('batch_number', $this->batch_number)->exists();
            if ($exists) {
                $this->addError('batch_number', 'This batch number already exists.');
            }
        }
    }

    /**
     * Real-time validation methods for stock intake
     */
    public function updatedBatchNumber()
    {
        $this->validateOnly('batch_number');
    }

    public function updatedReceivedDate()
    {
        $this->validateOnly('received_date');
    }

    public function updatedSupplierId()
    {
        $this->validateOnly('supplier_id');
    }

    public function updatedNotes()
    {
        $this->validateOnly('notes');
    }

    /**
     * Validate expiry date is after manufacture date
     */
    public function validateExpiryAfterManufacture($index)
    {
        if (!isset($this->batchItems[$index])) {
            return;
        }

        $batchItem = $this->batchItems[$index];
        
        if (!empty($batchItem['manufacture_date']) && !empty($batchItem['expiry_date'])) {
            $manufactureDate = Carbon::parse($batchItem['manufacture_date']);
            $expiryDate = Carbon::parse($batchItem['expiry_date']);
            
            if ($expiryDate->lte($manufactureDate)) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Validation Error!',
                    'message' => 'Expiry date must be after manufacture date for batch item ' . ($index + 1)
                ]);
                return false;
            }
        }
        
        return true;
    }

    /**
     * Validate stock out quantity against available stock
     */
    public function validateStockOutQuantity($batchItemId, $requestedQuantity)
    {
        $batchItem = \App\Models\BatchItem::find($batchItemId);
        if (!$batchItem) {
            return ['valid' => false, 'message' => 'Batch item not found.'];
        }

        $availableQuantity = $batchItem->current_quantity;
        
        // Convert requested quantity to base units if variant exists
        $baseRequestedQuantity = $requestedQuantity;
        if ($batchItem->variant) {
            try {
                $baseRequestedQuantity = $this->convertQuantityToBaseUnit(
                    $requestedQuantity, 
                    $batchItem->variant->unit_type
                );
            } catch (\Exception $e) {
                return ['valid' => false, 'message' => 'Invalid unit conversion: ' . $e->getMessage()];
            }
        }

        if ($baseRequestedQuantity > $availableQuantity) {
            $displayAvailable = $availableQuantity;
            if ($batchItem->variant) {
                $displayAvailable = UnitConverter::convertFromBase(
                    $availableQuantity, 
                    $batchItem->variant->unit_type
                );
            }
            
            return [
                'valid' => false, 
                'message' => 'Requested quantity exceeds available stock. Available: ' . 
                           number_format($displayAvailable, 3) . 
                           ($batchItem->variant ? ' ' . $batchItem->variant->unit_type : '')
            ];
        }

        return ['valid' => true, 'message' => ''];
    }

    /**
     * Enhanced expiry date validation for stock operations
     */
    public function validateExpiryForStockOut($batchId)
    {
        $batch = StockBatch::find($batchId);
        if (!$batch) {
            return ['valid' => false, 'message' => 'Batch not found.'];
        }

        // Check batch items for expiry
        $expiredItems = [];
        $expiringSoonItems = [];
        
        foreach ($batch->batchItems as $batchItem) {
            if ($batchItem->expiry_date) {
                if (Carbon::parse($batchItem->expiry_date)->isPast()) {
                    $expiredItems[] = $batchItem->inventoryItem->name;
                } elseif (Carbon::parse($batchItem->expiry_date)->diffInDays(now()) <= 7) {
                    $expiringSoonItems[] = [
                        'name' => $batchItem->inventoryItem->name,
                        'days' => Carbon::parse($batchItem->expiry_date)->diffInDays(now())
                    ];
                }
            }
        }

        if (!empty($expiredItems)) {
            return [
                'valid' => false,
                'type' => 'expired',
                'message' => 'The following items have expired: ' . implode(', ', $expiredItems) . 
                           '. Override approval is required.',
                'items' => $expiredItems
            ];
        }

        if (!empty($expiringSoonItems)) {
            $messages = [];
            foreach ($expiringSoonItems as $item) {
                $messages[] = $item['name'] . ' (expires in ' . $item['days'] . ' days)';
            }
            
            return [
                'valid' => true,
                'type' => 'expiring_soon',
                'message' => 'Warning: The following items are expiring soon: ' . implode(', ', $messages),
                'items' => $expiringSoonItems
            ];
        }

        return ['valid' => true, 'message' => ''];
    }

    /**
     * Comprehensive unit conversion validation
     */
    public function validateUnitConversion($value, $fromUnit, $toUnit = null)
    {
        try {
            // Validate the from unit
            if (!UnitConverter::isValidUnit($fromUnit)) {
                return ['valid' => false, 'message' => "Invalid unit: {$fromUnit}"];
            }

            // Validate the value
            if (!is_numeric($value) || $value < 0) {
                return ['valid' => false, 'message' => 'Value must be a positive number'];
            }

            // Test conversion to base unit
            $baseValue = UnitConverter::convertToBase($value, $fromUnit);
            
            // If target unit specified, test conversion
            if ($toUnit) {
                if (!UnitConverter::isValidUnit($toUnit)) {
                    return ['valid' => false, 'message' => "Invalid target unit: {$toUnit}"];
                }
                
                if (!UnitConverter::areUnitsCompatible($fromUnit, $toUnit)) {
                    return ['valid' => false, 'message' => "Units {$fromUnit} and {$toUnit} are not compatible"];
                }
                
                UnitConverter::convert($value, $fromUnit, $toUnit);
            }

            return ['valid' => true, 'message' => ''];
            
        } catch (\Exception $e) {
            return ['valid' => false, 'message' => 'Unit conversion error: ' . $e->getMessage()];
        }
    }

    public function convertQuantityToBaseUnit($quantity, $unit)
    {
        try {
            return UnitConverter::convertToBase($quantity, $unit);
        } catch (\Exception $e) {
            throw new \Exception('Invalid unit conversion: ' . $e->getMessage());
        }
    }

    public function saveStockIntake()
    {
        $this->validate();

        try {
            // Create stock batch (container for multiple items)
            $stockBatch = StockBatch::create([
                'batch_number' => $this->batch_number,
                'supplier_id' => $this->supplier_id,
                'received_date' => $this->received_date,
                'notes' => $this->notes,
                // These will be null for multi-item batches
                'inventory_item_id' => null,
                'item_variant_id' => null,
                'initial_quantity' => 0,
                'current_quantity' => 0,
                'unit_cost' => 0,
                'expiry_date' => null,
                'manufacture_date' => null,
            ]);

            $totalItems = 0;
            
            // Create batch items for each item in the batch
            foreach ($this->batchItems as $index => $batchItemData) {
                // Validate that inventory_item_id is not empty
                if (empty($batchItemData['inventory_item_id'])) {
                    throw new \Exception("Inventory item ID is required for batch item " . ($index + 1));
                }

                $item = InventoryItem::find($batchItemData['inventory_item_id']);
                if (!$item) {
                    throw new \Exception('Invalid inventory item selected for batch item ' . ($index + 1));
                }

                $variant = null;
                if (!empty($batchItemData['item_variant_id'])) {
                    $variant = ItemVariant::find($batchItemData['item_variant_id']);
                }
                
                // Validate required fields
                if (empty($batchItemData['initial_quantity']) || $batchItemData['initial_quantity'] <= 0) {
                    throw new \Exception('Initial quantity is required for batch item ' . ($index + 1));
                }
                
                if (!isset($batchItemData['unit_cost']) || $batchItemData['unit_cost'] < 0) {
                    throw new \Exception('Unit cost is required for batch item ' . ($index + 1));
                }
                
                // Convert quantity to base units for storage
                $baseQuantity = floatval($batchItemData['initial_quantity']);
                if ($variant) {
                    $baseQuantity = $this->convertQuantityToBaseUnit(
                        $batchItemData['initial_quantity'], 
                        $variant->unit_type
                    );
                }

                // Create batch item
                $batchItem = \App\Models\BatchItem::create([
                    'stock_batch_id' => $stockBatch->id,
                    'inventory_item_id' => intval($batchItemData['inventory_item_id']),
                    'item_variant_id' => !empty($batchItemData['item_variant_id']) ? intval($batchItemData['item_variant_id']) : null,
                    'initial_quantity' => $baseQuantity,
                    'current_quantity' => $baseQuantity,
                    'unit_cost' => floatval($batchItemData['unit_cost']),
                    'notes' => $batchItemData['notes'] ?? '',
                    'expiry_date' => !empty($batchItemData['expiry_date']) ? $batchItemData['expiry_date'] : null,
                    'manufacture_date' => !empty($batchItemData['manufacture_date']) ? $batchItemData['manufacture_date'] : null,
                ]);

                // Create stock transaction record for each item
                StockTransaction::create([
                    'batch_id' => $stockBatch->id,
                    'batch_item_id' => $batchItem->id,
                    'inventory_item_id' => intval($batchItemData['inventory_item_id']),
                    'item_variant_id' => !empty($batchItemData['item_variant_id']) ? intval($batchItemData['item_variant_id']) : null,
                    'transaction_type' => 'in',
                    'quantity' => $baseQuantity,
                    'remaining_quantity' => $baseQuantity,
                    'used_for' => 'Stock Intake',
                    'conducted_by' => auth()->user()->name ?? 'System',
                    'transaction_date' => $this->received_date,
                    'notes' => $batchItemData['notes'] ?? '',
                ]);
                
                $totalItems++;
            }

            $this->resetForm();
            $this->closeStockIntakeModal();
            
            // Toast notification for success
            $this->dispatch('toastMagic', [
                'status' => 'success',
                'title' => 'Success!',
                'message' => "Stock intake recorded successfully! Added {$totalItems} items to batch {$this->batch_number}"
            ]);
            
            $this->dispatch('stockIntakeCompleted');

        } catch (\Exception $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'Error recording stock intake: ' . $e->getMessage()
            ]);
        }
    }

    public function resetForm()
    {
        $this->reset([
            'batch_number',
            'supplier_id',
            'notes',
            'batchItems'
        ]);
        
        $this->received_date = Carbon::today()->format('Y-m-d');
    }

    public function resetStockOutForm()
    {
        $this->reset([
            'selected_batch_id',
            'out_quantity',
            'used_for',
            'approved_by',
            'out_notes',
            'override_expiry'
        ]);
        
        $this->transaction_date = Carbon::today()->format('Y-m-d');
        $this->conducted_by = auth()->user()->name ?? '';
    }

    public function openStockOutModal($batchId)
    {
        $this->selected_batch_id = $batchId;
        $this->showStockOutModal = true;
        
        // Load available items from this batch for stock out
        $batch = StockBatch::with(['batchItems.inventoryItem', 'batchItems.variant'])->find($batchId);
        $this->stockOutItems = [];
        
        if ($batch) {
            foreach ($batch->batchItems->where('current_quantity', '>', 0) as $batchItem) {
                $this->stockOutItems[] = [
                    'batch_item_id' => $batchItem->id,
                    'inventory_item_id' => $batchItem->inventory_item_id,
                    'item_variant_id' => $batchItem->item_variant_id,
                    'item_name' => $batchItem->inventoryItem->name,
                    'variant_name' => $batchItem->variant ? $batchItem->variant->display_name : null,
                    'available_quantity' => $batchItem->current_quantity,
                    'out_quantity' => '',
                    'selected' => false,
                ];
            }
        }
        
        // Reset form
        $this->reset([
            'used_for',
            'approved_by',
            'out_notes',
            'override_expiry'
        ]);
        
        $this->transaction_date = Carbon::today()->format('Y-m-d');
        $this->conducted_by = auth()->user()->name ?? '';
    }

    public function closeStockOutModal()
    {
        $this->showStockOutModal = false;
        $this->resetStockOutForm();
    }



    public function openTransactionHistoryModal($batchId)
    {
        $this->selected_batch_for_history = $batchId;
        $this->showTransactionHistoryModal = true;
    }

    public function closeTransactionHistoryModal()
    {
        $this->showTransactionHistoryModal = false;
        $this->selected_batch_for_history = null;
    }

    public function openBatchViewModal($batchId)
    {
        $this->selected_batch_for_view = $batchId;
        $this->showBatchViewModal = true;
        
        // Load batch items directly as collection (not array)
        $this->batchItemsForView = \App\Models\BatchItem::where('stock_batch_id', $batchId)
            ->with(['inventoryItem', 'variant'])
            ->get();
            
        // Debug: Log the results to see what we're getting
        \Log::info('Batch Items Loaded:', [
            'batch_id' => $batchId,
            'count' => $this->batchItemsForView->count(),
            'items' => $this->batchItemsForView->toArray()
        ]);
    }

    public function closeBatchViewModal()
    {
        $this->showBatchViewModal = false;
        $this->selected_batch_for_view = null;
        $this->batchItemsForView = [];
    }

    public function getBatchItemsProperty()
    {
        if (!$this->selected_batch_for_view) {
            return collect();
        }

        $batchItems = \App\Models\BatchItem::where('stock_batch_id', $this->selected_batch_for_view)
            ->with(['inventoryItem', 'variant'])
            ->get();
            
        // Debug: Log the results to see what we're getting
        \Log::info('Batch Items Query Result:', [
            'batch_id' => $this->selected_batch_for_view,
            'count' => $batchItems->count(),
            'items' => $batchItems->toArray()
        ]);
        
        return $batchItems;
    }

    public function getTransactionHistoryProperty()
    {
        if (!$this->selected_batch_for_history) {
            return collect();
        }

        return StockTransaction::where('batch_id', $this->selected_batch_for_history)
            ->with(['batchItem.inventoryItem', 'batchItem.variant'])
            ->orderBy('transaction_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();
    }

    protected function getStockOutRules()
    {
        $batch = StockBatch::find($this->selected_batch_id);
        
        $rules = [
            'selected_batch_id' => 'required|exists:stock_batches,id',
            'out_quantity' => 'required|numeric|min:0.01',
            'used_for' => 'required|string|max:255',
            'conducted_by' => 'required|string|max:255',
            'approved_by' => 'nullable|string|max:255',
            'transaction_date' => 'required|date|before_or_equal:today',
            'out_notes' => 'nullable|string|max:1000',
        ];

        // Add quantity validation based on available stock
        if ($batch) {
            $maxQuantity = $batch->current_quantity;
            if ($batch->variant) {
                $maxQuantity = UnitConverter::convertFromBase($batch->current_quantity, UnitConverter::getBaseUnit($batch->variant->unit_type));
            }
            $rules['out_quantity'] = 'required|numeric|min:0.01|max:' . $maxQuantity;
        }

        return $rules;
    }

    protected function getStockOutMessages()
    {
        $batch = StockBatch::find($this->selected_batch_id);
        $maxQuantity = 0;
        
        if ($batch) {
            $maxQuantity = $batch->current_quantity;
            if ($batch->variant) {
                $maxQuantity = UnitConverter::convertFromBase($batch->current_quantity, UnitConverter::getBaseUnit($batch->variant->unit_type));
            }
        }

        return [
            'selected_batch_id.required' => 'Batch selection is required.',
            'selected_batch_id.exists' => 'Selected batch is invalid.',
            'out_quantity.required' => 'Quantity is required.',
            'out_quantity.numeric' => 'Quantity must be a number.',
            'out_quantity.min' => 'Quantity must be greater than 0.',
            'out_quantity.max' => 'Quantity cannot exceed available stock (' . number_format($maxQuantity, 2) . ').',
            'used_for.required' => 'Usage purpose is required.',
            'used_for.max' => 'Usage purpose cannot exceed 255 characters.',
            'conducted_by.required' => 'Conducted by field is required.',
            'conducted_by.max' => 'Conducted by cannot exceed 255 characters.',
            'approved_by.max' => 'Approved by cannot exceed 255 characters.',
            'transaction_date.required' => 'Transaction date is required.',
            'transaction_date.date' => 'Please enter a valid transaction date.',
            'transaction_date.before_or_equal' => 'Transaction date cannot be in the future.',
            'out_notes.max' => 'Notes cannot exceed 1000 characters.',
        ];
    }

    public function checkExpiryWarning()
    {
        if (!$this->selected_batch_id) {
            return null;
        }

        $batch = StockBatch::find($this->selected_batch_id);
        if (!$batch || !$batch->expiry_date) {
            return null;
        }

        if ($batch->isExpired()) {
            return [
                'type' => 'expired',
                'message' => 'This batch expired on ' . $batch->expiry_date->format('M d, Y') . '. Usage requires approval override.',
                'days' => $batch->expiry_date->diffInDays(now(), false)
            ];
        }

        if ($batch->isExpiringSoon(7)) {
            return [
                'type' => 'expiring_soon',
                'message' => 'This batch expires in ' . $batch->expiry_date->diffInDays(now()) . ' days (' . $batch->expiry_date->format('M d, Y') . ').',
                'days' => $batch->expiry_date->diffInDays(now())
            ];
        }

        return null;
    }

    public function saveStockOut()
    {
        // Validate that at least one item is selected
        $selectedItems = collect($this->stockOutItems)->where('selected', true);
        if ($selectedItems->isEmpty()) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Validation Error!',
                'message' => 'Please select at least one item to stock out.'
            ]);
            return;
        }

        // Validate each selected item
        foreach ($selectedItems as $index => $item) {
            if (empty($item['out_quantity']) || $item['out_quantity'] <= 0) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Validation Error!',
                    'message' => 'Please enter a valid quantity for all selected items.'
                ]);
                return;
            }
            
            if ($item['out_quantity'] > $item['available_quantity']) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Validation Error!',
                    'message' => 'Quantity for ' . $item['item_name'] . ' exceeds available stock.'
                ]);
                return;
            }
        }

        // Validate common fields
        if (empty($this->used_for) || empty($this->conducted_by) || empty($this->transaction_date)) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Validation Error!',
                'message' => 'Please fill in all required fields.'
            ]);
            return;
        }

        try {
            $batch = StockBatch::find($this->selected_batch_id);
            if (!$batch) {
                throw new \Exception('Selected batch not found.');
            }

            // Check expiry and require override if expired
            $expiryWarning = $this->checkExpiryWarning();
            if ($expiryWarning && $expiryWarning['type'] === 'expired' && !$this->override_expiry) {
                // Toast notification for warning
                $this->dispatch('toastMagic', [
                    'status' => 'warning',
                    'title' => 'Expired Item Warning!',
                    'message' => $expiryWarning['message'] . ' Please check the override option to proceed.'
                ]);
                
                $this->addError('override_expiry', 'This batch is expired. Please check the override option to proceed.');
                return;
            }

            $totalItemsProcessed = 0;
            $totalQuantityUsed = 0;

            // Process each selected item
            foreach ($selectedItems as $stockOutItem) {
                $batchItem = \App\Models\BatchItem::find($stockOutItem['batch_item_id']);
                if (!$batchItem) {
                    continue;
                }

                // Convert quantity to base units for storage
                $baseQuantity = $stockOutItem['out_quantity'];
                if ($batchItem->variant) {
                    $baseQuantity = $this->convertQuantityToBaseUnit(
                        $stockOutItem['out_quantity'], 
                        $batchItem->variant->unit_type
                    );
                }

                // Update batch item quantity
                $newQuantity = $batchItem->current_quantity - $baseQuantity;
                $batchItem->update(['current_quantity' => max(0, $newQuantity)]);

                // Create stock transaction record
                StockTransaction::create([
                    'batch_id' => $batch->id,
                    'batch_item_id' => $batchItem->id,
                    'inventory_item_id' => $batchItem->inventory_item_id,
                    'item_variant_id' => $batchItem->item_variant_id,
                    'transaction_type' => 'out',
                    'quantity' => $baseQuantity,
                    'remaining_quantity' => max(0, $newQuantity),
                    'used_for' => $this->used_for,
                    'conducted_by' => $this->conducted_by,
                    'approved_by' => $this->approved_by,
                    'transaction_date' => $this->transaction_date,
                    'notes' => $this->out_notes,
                ]);

                $totalItemsProcessed++;
                $totalQuantityUsed += $baseQuantity;
            }

            $this->closeStockOutModal();
            
            // Toast notification for success
            $this->dispatch('toastMagic', [
                'status' => 'success',
                'title' => 'Success!',
                'message' => "Stock out recorded successfully! Processed {$totalItemsProcessed} items from batch {$batch->batch_number}"
            ]);
            
            $this->dispatch('stockOutCompleted');

        } catch (\Exception $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'Error recording stock out: ' . $e->getMessage()
            ]);
        }
    }

    public function exportTransactions()
    {
        $query = StockTransaction::query()
            ->with(['inventoryItem.category', 'batch.supplier'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('used_for', 'like', '%' . $this->search . '%')
                      ->orWhere('conducted_by', 'like', '%' . $this->search . '%')
                      ->orWhereHas('inventoryItem', function($itemQuery) {
                          $itemQuery->where('name', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->itemFilter, function($query) {
                $query->where('inventory_item_id', $this->itemFilter);
            })
            ->when($this->supplierFilter, function($query) {
                $query->whereHas('batch', function($batchQuery) {
                    $batchQuery->where('supplier_id', $this->supplierFilter);
                });
            })
            ->when($this->transactionTypeFilter, function($query) {
                $query->where('transaction_type', $this->transactionTypeFilter);
            })
            ->when($this->dateFrom, function($query) {
                $query->where('transaction_date', '>=', $this->dateFrom);
            })
            ->when($this->dateTo, function($query) {
                $query->where('transaction_date', '<=', $this->dateTo);
            });

        return Excel::download(new StockTransactionExport($query), 'stock_transactions.xlsx');
    }

    public function exportTransactionsPdf()
    {
        $data = StockTransaction::query()
            ->with(['inventoryItem.category', 'batch.supplier'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('used_for', 'like', '%' . $this->search . '%')
                      ->orWhere('conducted_by', 'like', '%' . $this->search . '%')
                      ->orWhereHas('inventoryItem', function($itemQuery) {
                          $itemQuery->where('name', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->itemFilter, function($query) {
                $query->where('inventory_item_id', $this->itemFilter);
            })
            ->when($this->supplierFilter, function($query) {
                $query->whereHas('batch', function($batchQuery) {
                    $batchQuery->where('supplier_id', $this->supplierFilter);
                });
            })
            ->when($this->transactionTypeFilter, function($query) {
                $query->where('transaction_type', $this->transactionTypeFilter);
            })
            ->when($this->dateFrom, function($query) {
                $query->where('transaction_date', '>=', $this->dateFrom);
            })
            ->when($this->dateTo, function($query) {
                $query->where('transaction_date', '<=', $this->dateTo);
            })
            ->orderBy('transaction_date', 'desc')
            ->get();

        $pdf = Pdf::loadView('exports.stock-transaction-pdf', [
            'data' => $data,
            'title' => 'Stock Transactions Report',
            'generatedAt' => now()->format('Y-m-d H:i:s')
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'stock_transactions_report_' . now()->format('Y-m-d_H-i-s') . '.pdf');
    }

    public function render()
    {
        $stockBatches = StockBatch::query()
            ->with(['batchItems.inventoryItem', 'batchItems.variant', 'supplier'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('batch_number', 'like', '%' . $this->search . '%')
                      ->orWhereHas('batchItems.inventoryItem', function($itemQuery) {
                          $itemQuery->where('name', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('supplier', function($supplierQuery) {
                          $supplierQuery->where('name', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->itemFilter, function($query) {
                $query->whereHas('batchItems', function($batchItemQuery) use ($query) {
                    $batchItemQuery->where('inventory_item_id', $this->itemFilter);
                });
            })
            ->when($this->supplierFilter, function($query) {
                $query->where('supplier_id', $this->supplierFilter);
            })
            ->when($this->statusFilter, function($query) {
                if ($this->statusFilter === 'expired') {
                    $query->whereHas('batchItems', function($batchItemQuery) {
                        $batchItemQuery->where('expiry_date', '<', Carbon::now());
                    });
                } elseif ($this->statusFilter === 'expiring_soon') {
                    $query->whereHas('batchItems', function($batchItemQuery) {
                        $batchItemQuery->where('expiry_date', '>=', Carbon::now())
                                      ->where('expiry_date', '<=', Carbon::now()->addDays(30));
                    });
                } elseif ($this->statusFilter === 'low_stock') {
                    $query->whereHas('batchItems', function($batchItemQuery) {
                        $batchItemQuery->whereColumn('current_quantity', '<=', \DB::raw('initial_quantity * 0.2'))
                                      ->where('current_quantity', '>', 0);
                    });
                } elseif ($this->statusFilter === 'out_of_stock') {
                    $query->whereHas('batchItems', function($batchItemQuery) {
                        $batchItemQuery->where('current_quantity', '<=', 0);
                    });
                }
            })
            ->when($this->dateFrom, function($query) {
                $query->where('received_date', '>=', $this->dateFrom);
            })
            ->when($this->dateTo, function($query) {
                $query->where('received_date', '<=', $this->dateTo);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        $inventoryItems = InventoryItem::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();

        return view('livewire.inventory.stock-management', [
            'stockBatches' => $stockBatches,
            'inventoryItems' => $inventoryItems,
            'suppliers' => $suppliers
        ]);
    }
}