<?php

namespace App\Livewire\Inventory;

use App\Models\ItemsSlot;
use App\Models\Item;
use App\Models\ItemFile;
use App\Models\SlotFile;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class SlotItems extends Component
{
    use WithFileUploads;

    public $itemsSlotId = null;
    public $itemsSlot = null;

    // Form fields
    public $item_name;
    public $item_info;
    public $item_photo;
    public $item_files = [];
    public $slot_files = [];
    public $file_names = [];
    public $upload_type = 'item'; // 'item' or 'file'

    public $showAddNewModal = false;
    public $isEditing = false;
    public $itemId = null;

    protected $listeners = [
        'editItem' => 'loadItemData',
        'resetForm' => 'resetForm'
    ];

    public function mount($itemsSlotId)
    {
        $this->itemsSlotId = $itemsSlotId;
        $this->itemsSlot = ItemsSlot::with(['subBox.box.lab', 'user'])->findOrFail($itemsSlotId);

        // Initialize arrays
        $this->slot_files = [null];
        $this->file_names = [''];
    }

    protected function rules()
    {
        $rules = [];

        if ($this->upload_type === 'item') {
            $rules = [
                'item_name' => 'required|string|max:255',
                'item_info' => 'nullable|string',
                'item_photo' => 'nullable|image|max:2048',
                'item_files.*' => 'nullable|file|max:2048',
            ];
        } else {
            $rules = [
                'slot_files.*' => 'required|file|max:2048',
                'file_names.*' => 'required|string|max:255',
            ];
        }

        return $rules;
    }

    public function toggleShowAddNewModal()
    {
        if (!$this->itemsSlot->status) {
            session()->flash('error', 'Cannot add items to inactive slot!');
            return;
        }

        $this->showAddNewModal = true;
    }

    public function closeAddNewModal()
    {
        $this->showAddNewModal = false;
        $this->resetForm();
    }

    public function addFileInput()
    {
        $this->slot_files[] = null;
        $this->file_names[] = '';
    }

    public function removeFileInput($index)
    {
        unset($this->slot_files[$index]);
        unset($this->file_names[$index]);
        $this->slot_files = array_values($this->slot_files);
        $this->file_names = array_values($this->file_names);
    }

    public function loadItemData($itemId)
    {
        $this->itemId = $itemId;
        $this->isEditing = true;
        $item = Item::findOrFail($itemId);

        $this->item_name = $item->item_name;
        $this->item_info = $item->item_info;
        $this->upload_type = 'item';
        $this->showAddNewModal = true;
    }

    public function save()
    {
        if ($this->upload_type === 'item') {
            $this->saveItem();
        } else {
            $this->saveFiles();
        }
    }

    public function saveItem()
    {
        // Check capacity for new items
        if (!$this->isEditing && $this->itemsSlot->available_capacity <= 0) {
            session()->flash('error', 'Slot capacity is full! Cannot add more items.');
            return;
        }

        $this->validate();

        if ($this->isEditing) {
            $this->updateItem();
        } else {
            $this->createItem();
        }
    }

    public function createItem()
    {
        $item = new Item();
        $item->items_slot_id = $this->itemsSlotId;
        $item->item_name = $this->item_name;
        $item->item_info = $this->item_info;

        // Handle item photo upload
        if ($this->item_photo) {
            $photoPath = $this->item_photo->store('item-photos', 'public');
            $item->item_photo = $photoPath;
        }
        $item->save();


        foreach ($this->item_files as $file) {
            try {
                if ($file) {
                    $filePath = $file->store('item-files', 'public');

                    ItemFile::create([
                        'item_id' => $item->id,
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $filePath,
                        'file_type' => $file->getClientMimeType(),
                        'file_size' => $file->getSize(),
                    ]);
                }
            } catch (\Throwable $e) {
                // Log the error with file name if available
                $fileName = $file ? $file->getClientOriginalName() : 'Unknown File';
                Log::error('Error saving item file: ' . $fileName, [
                    'message' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);
            }
        }

        $this->resetForm();
        $this->closeAddNewModal();
        session()->flash('success', 'Item created successfully!');

        $this->dispatch('itemCreated');
    }

    public function updateItem()
    {
        $item = Item::findOrFail($this->itemId);
        $item->item_name = $this->item_name;
        $item->item_info = $this->item_info;

        // Handle item photo upload
        if ($this->item_photo) {
            // Delete old photo if exists
            if ($item->item_photo) {
                Storage::disk('public')->delete($item->item_photo);
            }
            $photoPath = $this->item_photo->store('item-photos', 'public');
            $item->item_photo = $photoPath;
        }

        $item->save();

        // Handle new item files upload
        if ($this->item_files) {
            foreach ($this->item_files as $file) {
                if ($file) {
                    $filePath = $file->store('item-files', 'public');
                    ItemFile::create([
                        'item_id' => $item->id,
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $filePath,
                        'file_type' => $file->getClientMimeType(),
                        'file_size' => $file->getSize(),
                    ]);
                }
            }
        }

        $this->resetForm();
        $this->closeAddNewModal();
        session()->flash('success', 'Item updated successfully!');

        $this->dispatch('itemUpdated');
    }

    public function saveFiles()
    {
        $this->validate();

        foreach ($this->slot_files as $index => $file) {
            if ($file && isset($this->file_names[$index])) {
                $filePath = $file->store('slot-files', 'public');
                SlotFile::create([
                    'items_slot_id' => $this->itemsSlotId,
                    'file_name' => $this->file_names[$index],
                    'file_path' => $filePath,
                    'file_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        $this->resetForm();
        $this->closeAddNewModal();
        session()->flash('success', 'Files uploaded successfully!');

        $this->dispatch('filesUploaded');
    }

    public function deleteItem($itemId)
    {
        $item = Item::findOrFail($itemId);

        // Delete item photo
        if ($item->item_photo) {
            Storage::disk('public')->delete($item->item_photo);
        }

        // Delete item files
        foreach ($item->itemFiles as $itemFile) {
            Storage::disk('public')->delete($itemFile->file_path);
        }

        $item->delete();

        session()->flash('success', 'Item deleted successfully!');
        $this->dispatch('itemDeleted');
    }

    public function deleteSlotFile($fileId)
    {
        $file = SlotFile::findOrFail($fileId);
        Storage::disk('public')->delete($file->file_path);
        $file->delete();

        session()->flash('success', 'File deleted successfully!');
        $this->dispatch('fileDeleted');
    }

    public function deleteItemFile($fileId)
    {
        $file = ItemFile::findOrFail($fileId);
        Storage::disk('public')->delete($file->file_path);
        $file->delete();

        session()->flash('success', 'File deleted successfully!');
        $this->dispatch('fileDeleted');
    }

    public function viewFile($fileType, $fileId)
    {
        if ($fileType === 'item') {
            $file = ItemFile::findOrFail($fileId);
        } else {
            $file = SlotFile::findOrFail($fileId);
        }

        $filePath = Storage::disk('public')->path($file->file_path);
        $mimeType = $file->file_type;

        // Check if file is image
        if (str_starts_with($mimeType, 'image/')) {
            $this->dispatch('showImageModal', [
                'url' => Storage::url($file->file_path),
                'name' => $file->file_name
            ]);
        }
        // Check if file is PDF
        elseif ($mimeType === 'application/pdf') {
            $this->dispatch('showPdfModal', [
                'url' => Storage::url($file->file_path),
                'name' => $file->file_name
            ]);
        }
        // For other file types, trigger download
        else {
            $this->dispatch('downloadFile', [
                'url' => Storage::url($file->file_path),
                'name' => $file->file_name
            ]);
        }
    }

    public function resetForm()
    {
        $this->reset([
            'item_name',
            'item_info',
            'item_photo',
            'item_files',
            'slot_files',
            'file_names',
            'isEditing',
            'itemId'
        ]);

        $this->upload_type = 'item';
        $this->slot_files = [null];
        $this->file_names = [''];
    }

    public function render()
    {
        $items = Item::where('items_slot_id', $this->itemsSlotId)
                    ->with('itemFiles')
                    ->orderBy('created_at', 'desc')
                    ->get();

        $slotFiles = SlotFile::where('items_slot_id', $this->itemsSlotId)
                            ->orderBy('created_at', 'desc')
                            ->get();

        return view('livewire.inventory.slot-items', [
            'items' => $items,
            'slotFiles' => $slotFiles
        ]);
    }
}
