<?php

namespace App\Livewire\Inventory;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\StockBatch;
use App\Models\BatchItem;
use App\Models\StockTransaction;
use App\Models\InventoryItem;
use App\Models\Category;
use App\Exports\ExpiryReportExport;
use App\Exports\ConsumptionReportExport;
use App\Exports\LowStockReportExport;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class Reports extends Component
{
    use WithPagination;

    public $activeTab = 'expiry';
    public $dateFrom;
    public $dateTo;
    public $categoryFilter = '';
    public $itemTypeFilter = '';
    public $expiryDays = 30;
    public $search = '';

    protected $paginationTheme = 'tailwind';

    public function mount()
    {
        $this->dateFrom = now()->subMonth()->format('Y-m-d');
        $this->dateTo = now()->format('Y-m-d');
    }

    public function updatedActiveTab()
    {
        $this->resetPage();
    }

    public function updatedDateFrom()
    {
        $this->resetPage();
    }

    public function updatedDateTo()
    {
        $this->resetPage();
    }

    public function updatedCategoryFilter()
    {
        $this->resetPage();
    }

    public function updatedItemTypeFilter()
    {
        $this->resetPage();
    }

    public function updatedExpiryDays()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function getExpiryReportQuery()
    {
        $query = BatchItem::with(['inventoryItem.category', 'stockBatch.supplier'])
            ->whereHas('inventoryItem')
            ->whereNotNull('expiry_date')
            ->where('current_quantity', '>', 0);

        // Filter by expiry date range
        $query->where('expiry_date', '<=', now()->addDays($this->expiryDays));

        // Apply filters
        if ($this->categoryFilter) {
            $query->whereHas('inventoryItem', function($q) {
                $q->where('category_id', $this->categoryFilter);
            });
        }

        if ($this->itemTypeFilter) {
            $query->whereHas('inventoryItem', function($q) {
                $q->where('item_type', $this->itemTypeFilter);
            });
        }

        if ($this->search) {
            $query->where(function($q) {
                $q->whereHas('stockBatch', function($subQ) {
                    $subQ->where('batch_number', 'like', '%' . $this->search . '%');
                  })
                  ->orWhereHas('inventoryItem', function($subQ) {
                      $subQ->where('name', 'like', '%' . $this->search . '%');
                  });
            });
        }

        return $query->orderBy('expiry_date', 'asc');
    }

    public function getConsumptionReportQuery()
    {
        $query = StockTransaction::with(['inventoryItem.category', 'batch'])
            ->where('transaction_type', 'out')
            ->whereBetween('transaction_date', [$this->dateFrom, $this->dateTo]);

        // Apply filters
        if ($this->categoryFilter) {
            $query->whereHas('inventoryItem', function($q) {
                $q->where('category_id', $this->categoryFilter);
            });
        }

        if ($this->itemTypeFilter) {
            $query->whereHas('inventoryItem', function($q) {
                $q->where('item_type', $this->itemTypeFilter);
            });
        }

        if ($this->search) {
            $query->where(function($q) {
                $q->where('used_for', 'like', '%' . $this->search . '%')
                  ->orWhere('conducted_by', 'like', '%' . $this->search . '%')
                  ->orWhereHas('inventoryItem', function($subQ) {
                      $subQ->where('name', 'like', '%' . $this->search . '%');
                  });
            });
        }

        return $query->orderBy('transaction_date', 'desc');
    }

    public function getLowStockReportQuery()
    {
        $query = InventoryItem::with(['category', 'stockBatches'])
            ->whereNotNull('min_stock_level')
            ->where('min_stock_level', '>', 0);

        // Apply filters
        if ($this->categoryFilter) {
            $query->where('category_id', $this->categoryFilter);
        }

        if ($this->itemTypeFilter) {
            $query->where('item_type', $this->itemTypeFilter);
        }

        if ($this->search) {
            $query->where('name', 'like', '%' . $this->search . '%');
        }

        // Filter items that are actually low on stock
        $query->whereHas('batchItems', function($q) {
            $q->selectRaw('inventory_item_id, SUM(current_quantity) as total_stock')
              ->groupBy('inventory_item_id')
              ->havingRaw('total_stock <= (SELECT min_stock_level FROM inventory_items WHERE id = inventory_item_id)');
        });

        return $query->orderBy('name');
    }

    public function exportExcel($reportType)
    {
        $filename = $reportType . '_report_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
        
        switch ($reportType) {
            case 'expiry':
                return Excel::download(new ExpiryReportExport($this->getExpiryReportQuery()), $filename);
            case 'consumption':
                return Excel::download(new ConsumptionReportExport($this->getConsumptionReportQuery()), $filename);
            case 'low_stock':
                return Excel::download(new LowStockReportExport($this->getLowStockReportQuery()), $filename);
        }
    }

    public function exportPdf($reportType)
    {
        $data = [];
        $title = '';
        
        switch ($reportType) {
            case 'expiry':
                $data = $this->getExpiryReportQuery()->get();
                $title = 'Expiry Report';
                break;
            case 'consumption':
                $data = $this->getConsumptionReportQuery()->get();
                $title = 'Consumption Report';
                break;
            case 'low_stock':
                $data = $this->getLowStockReportQuery()->get();
                $title = 'Low Stock Report';
                break;
        }

        $pdf = Pdf::loadView('reports.pdf.' . $reportType, [
            'data' => $data,
            'title' => $title,
            'dateFrom' => $this->dateFrom,
            'dateTo' => $this->dateTo,
            'generatedAt' => now()->format('Y-m-d H:i:s')
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, $reportType . '_report_' . now()->format('Y-m-d_H-i-s') . '.pdf');
    }

    public function render()
    {
        $categories = Category::orderBy('name')->get();
        
        $data = [];
        switch ($this->activeTab) {
            case 'expiry':
                $data = $this->getExpiryReportQuery()->paginate(10);
                break;
            case 'consumption':
                $data = $this->getConsumptionReportQuery()->paginate(10);
                break;
            case 'low_stock':
                $data = $this->getLowStockReportQuery()->paginate(10);
                break;
        }

        return view('livewire.inventory.reports', [
            'data' => $data,
            'categories' => $categories
        ]);
    }
}