<?php

namespace App\Livewire\Inventory;

use App\Models\ItemsSlot;
use App\Models\SubBox;
use App\Models\User;
use App\Models\PrincipalInvestigator;
use App\Models\Student;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Barryvdh\DomPDF\Facade\Pdf;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;

class ItemsSlots extends Component
{
    public $userId = null;
    public $subBoxId = null;
    public $user = null;
    public $subBox = null;
    public $items_slot_name;
    public $items_capacity;
    public $selected_user_id;
    public $status = 1;

    public $showAddNewItemsSlotModel = false;
    public $isEditing = false;
    public $itemsSlotId = null;
    public $availableUsers = [];
    
    // QR Code properties
    public $showQrCodeModal = false;
    public $qrCodeData;
    public $qrCodeCount = 1;
    public $selectedItemsSlot;

    protected $listeners = [
        'editItemsSlot' => 'loadItemsSlotData',
        'resetForm' => 'resetForm'
    ];

    public function mount($userId = null, $subBoxId = null)
    {
        $this->userId = $userId;
        $this->subBoxId = $subBoxId;

        if ($userId) {
            $this->user = User::findOrFail($userId);
        }

        if ($subBoxId) {
            $this->subBox = SubBox::with('box')->findOrFail($subBoxId);
        }

        $this->loadAvailableUsers();
    }

    protected function rules()
    {
        return [
            'items_slot_name' => 'required|string|max:255',
            'items_capacity' => 'required|integer|min:1',
            'selected_user_id' => 'nullable|exists:users,id',
        ];
    }

    public function loadAvailableUsers()
    {
        $currentUser = Auth::user();

        if ($currentUser->hasRole('super_admin')) {
            // Super admin can see all PIs and Students
            $piUsers = User::whereHas('principalInvestigators')->with('principalInvestigators')->get();
            $studentUsers = User::whereHas('students')->with('students')->get();

            $this->availableUsers = $piUsers->merge($studentUsers)->map(function ($user) {
                $profile = $user->principalInvestigators->first() ?? $user->students->first();
                return [
                    'id' => $user->id,
                    'name' => $profile ? $profile->first_name . ' ' . $profile->last_name : $user->name,
                    'type' => $user->principalInvestigators->count() > 0 ? 'PI' : 'Student',
                    'email' => $user->email
                ];
            });
        } elseif ($currentUser->hasRole('pi')) {
            // PI can see only their students
            $pi = $currentUser->principalInvestigators->first();
            if ($pi) {
                $studentUsers = User::whereHas('students', function($query) use ($pi) {
                    $query->where('principal_investigator_id', $pi->id);
                })->with('students')->get();

                $this->availableUsers = $studentUsers->map(function ($user) {
                    $student = $user->students->first();
                    return [
                        'id' => $user->id,
                        'name' => $student ? $student->first_name . ' ' . $student->last_name : $user->name,
                        'type' => 'Student',
                        'email' => $user->email
                    ];
                });
            }
        }
    }

    public function toggleShowAddNewItemsSlotModel()
    {
        $this->showAddNewItemsSlotModel = true;
    }

    public function closeAddNewItemsSlotModel()
    {
        $this->showAddNewItemsSlotModel = false;
        $this->resetForm();
    }

    public function loadItemsSlotData($itemsSlotId)
    {
        $this->itemsSlotId = $itemsSlotId;
        $this->isEditing = true;
        $itemsSlot = ItemsSlot::findOrFail($itemsSlotId);

        $this->items_slot_name = $itemsSlot->items_slot_name;
        $this->items_capacity = $itemsSlot->items_capacity;
        $this->selected_user_id = $itemsSlot->user_id;
        $this->status = $itemsSlot->status;
        $this->showAddNewItemsSlotModel = true;
    }

    public function toggleStatus($itemsSlotId)
    {
        $itemsSlot = ItemsSlot::findOrFail($itemsSlotId);
        $itemsSlot->status = !$itemsSlot->status;
        $itemsSlot->save();

        session()->flash('success', 'Items slot status updated successfully!');
    }

    public function saveItemsSlot()
    {
        $this->validate();

        if ($this->isEditing) {
            $this->updateItemsSlot();
        } else {
            $this->createItemsSlot();
        }
    }

    public function createItemsSlot()
    {
        $itemsSlot = new ItemsSlot();
        $itemsSlot->sub_box_id = $this->subBoxId;
        $itemsSlot->user_id = $this->selected_user_id;
        $itemsSlot->items_slot_name = $this->items_slot_name;
        $itemsSlot->items_capacity = $this->items_capacity;
        $itemsSlot->status = $this->status;
        $itemsSlot->save();

        $this->resetForm();
        $this->closeAddNewItemsSlotModel();
        session()->flash('success', 'Items slot created successfully!');

        $this->dispatch('itemsSlotCreated');
    }

    public function updateItemsSlot()
    {
        $itemsSlot = ItemsSlot::findOrFail($this->itemsSlotId);
        $itemsSlot->items_slot_name = $this->items_slot_name;
        $itemsSlot->items_capacity = $this->items_capacity;
        $itemsSlot->user_id = $this->selected_user_id;
        $itemsSlot->status = $this->status;
        $itemsSlot->save();

        $this->resetForm();
        $this->closeAddNewItemsSlotModel();
        session()->flash('success', 'Items slot updated successfully!');

        $this->dispatch('itemsSlotUpdated');
    }

    public function deleteItemsSlot($itemsSlotId)
    {
        $itemsSlot = ItemsSlot::findOrFail($itemsSlotId);
        $itemsSlot->delete();

        session()->flash('success', 'Items slot deleted successfully!');
        $this->dispatch('itemsSlotDeleted');
    }

    public function generateQrCode($itemsSlotId)
    {
        $itemsSlot = ItemsSlot::with(['user', 'subBox.box.lab', 'items', 'slotFiles'])
                              ->findOrFail($itemsSlotId);

        // Generate URL with encoded ID
        $url = route('inventory.items-slot-info', encode_id($itemsSlotId));

        // Generate QR code
        $rendererStyle = new RendererStyle(400);
        $imageRenderer = new ImageRenderer($rendererStyle, new SvgImageBackEnd());
        $writer = new Writer($imageRenderer);
        $svgContent = $writer->writeString($url);

        $this->qrCodeData = 'data:image/svg+xml;base64,' . base64_encode($svgContent);
        $this->selectedItemsSlot = $itemsSlot;
        $this->showQrCodeModal = true;
    }

    public function printQrCodes($itemsSlotId)
    {
        $this->validate([
            'qrCodeCount' => 'required|numeric|min:1|max:100',
        ]);

        $itemsSlot = ItemsSlot::with(['user', 'subBox.box.lab', 'items', 'slotFiles'])
                              ->findOrFail($itemsSlotId);
        $url = route('inventory.items-slot-info', encode_id($itemsSlotId));

        // Generate QR code data for PDF
        $rendererStyle = new RendererStyle(400);
        $imageRenderer = new ImageRenderer($rendererStyle, new SvgImageBackEnd());
        $writer = new Writer($imageRenderer);
        $svgContent = $writer->writeString($url);

        $qrCodeData = 'data:image/svg+xml;base64,' . base64_encode($svgContent);

        // Generate PDF
        $pdf = PDF::loadView('livewire.inventory.items-slot-qr-code-pdf', [
            'qrCodeData' => $qrCodeData,
            'count' => $this->qrCodeCount,
            'itemsSlot' => $itemsSlot,
        ]);

        $filename = "items-slot-qr-codes-{$itemsSlot->items_slot_name}-{$itemsSlot->id}.pdf";
        $filename = Str::slug($filename);

        return response()->streamDownload(
            fn () => print($pdf->output()),
            $filename,
            [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="'.$filename.'"',
            ]
        );
    }

    public function resetForm()
    {
        $this->reset([
            'items_slot_name',
            'items_capacity',
            'selected_user_id',
            'isEditing',
            'itemsSlotId'
        ]);

        $this->status = 1;
    }

    public function render()
    {
        $itemsSlots = collect();

        if ($this->userId) {
            $itemsSlots = ItemsSlot::where('user_id', $this->userId)
                                  ->with(['subBox.box', 'user'])
                                  ->withCount(['items', 'slotFiles'])
                                  ->orderBy('created_at', 'desc')
                                  ->get();
        } elseif ($this->subBoxId) {
            $itemsSlots = ItemsSlot::where('sub_box_id', $this->subBoxId)
                                  ->with(['subBox.box', 'user'])
                                  ->withCount(['items', 'slotFiles'])
                                  ->orderBy('created_at', 'desc')
                                  ->get();
        }

        return view('livewire.inventory.items-slots', [
            'itemsSlots' => $itemsSlots
        ]);
    }
}
