<?php

namespace App\Livewire\Inventory;

use App\Models\InventoryItem;
use App\Models\Category;
use App\Models\ItemVariant;
use App\Exports\ItemExport;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class ItemList extends Component
{
    use WithPagination;

    public $name;
    public $description;
    public $category_id;
    public $item_type = 'regular';
    public $min_stock_level;
    public $storage_conditions;
    public $hazard_information;
    
    // Variant fields for chemicals
    public $variants = [];
    public $newVariant = [
        'volume_per_unit' => '',
        'unit_type' => 'ml',
        'barcode' => ''
    ];
    
    public $search = '';
    public $categoryFilter = '';
    public $itemTypeFilter = '';
    public $showAddNewItemModal = false;
    public $isEditing = false;
    public $itemId = null;

    protected $listeners = [
        'editItem' => 'loadItemData',
        'resetForm' => 'resetForm'
    ];

    protected function rules()
    {
        $rules = [
            'name' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\s\-_\.\(\)]+$/',
                'unique:inventory_items,name,' . ($this->itemId ?? 'NULL')
            ],
            'description' => 'nullable|string|max:1000',
            'category_id' => 'required|exists:categories,id',
            'item_type' => 'required|in:regular,chemical',
            'min_stock_level' => 'nullable|integer|min:0|max:999999',
        ];

        if ($this->item_type === 'chemical') {
            $rules['storage_conditions'] = 'nullable|string|max:255';
            $rules['hazard_information'] = 'nullable|string|max:1000';
            
            // Validate that chemical items have at least one variant
            if (empty($this->variants)) {
                $rules['variants'] = 'required|array|min:1';
            } else {
                $rules['variants'] = 'required|array|min:1';
                $rules['variants.*.volume_per_unit'] = [
                    'required',
                    'numeric',
                    'min:0.001',
                    'max:999999.999'
                ];
                $rules['variants.*.unit_type'] = 'required|in:ml,mg,g,L,kg';
                $rules['variants.*.barcode'] = [
                    'nullable',
                    'string',
                    'max:255',
                    'regex:/^[a-zA-Z0-9\-_]+$/'
                ];
                
                // Add custom validation for barcode uniqueness
                foreach ($this->variants as $index => $variant) {
                    if (!empty($variant['barcode'])) {
                        $rules["variants.{$index}.barcode"][] = 'unique:item_variants,barcode' . 
                            (isset($variant['id']) ? ',' . $variant['id'] : '');
                    }
                }
            }
        }

        return $rules;
    }

    protected $messages = [
        'name.required' => 'Item name is required.',
        'name.string' => 'Item name must be valid text.',
        'name.max' => 'Item name cannot exceed 255 characters.',
        'name.regex' => 'Item name can only contain letters, numbers, spaces, hyphens, underscores, periods, and parentheses.',
        'name.unique' => 'An item with this name already exists.',
        'description.string' => 'Description must be valid text.',
        'description.max' => 'Description cannot exceed 1000 characters.',
        'category_id.required' => 'Please select a category.',
        'category_id.exists' => 'Selected category is invalid.',
        'item_type.required' => 'Please select an item type.',
        'item_type.in' => 'Item type must be either regular or chemical.',
        'min_stock_level.integer' => 'Minimum stock level must be a whole number.',
        'min_stock_level.min' => 'Minimum stock level cannot be negative.',
        'min_stock_level.max' => 'Minimum stock level cannot exceed 999,999.',
        'storage_conditions.string' => 'Storage conditions must be valid text.',
        'storage_conditions.max' => 'Storage conditions cannot exceed 255 characters.',
        'hazard_information.string' => 'Hazard information must be valid text.',
        'hazard_information.max' => 'Hazard information cannot exceed 1000 characters.',
        'variants.required' => 'Chemical items must have at least one variant.',
        'variants.array' => 'Variants must be a valid list.',
        'variants.min' => 'Chemical items must have at least one variant.',
        'variants.*.volume_per_unit.required' => 'Volume per unit is required for all variants.',
        'variants.*.volume_per_unit.numeric' => 'Volume per unit must be a valid number.',
        'variants.*.volume_per_unit.min' => 'Volume per unit must be at least 0.001.',
        'variants.*.volume_per_unit.max' => 'Volume per unit cannot exceed 999,999.999.',
        'variants.*.unit_type.required' => 'Unit type is required for all variants.',
        'variants.*.unit_type.in' => 'Unit type must be one of: ml, L, mg, g, kg.',
        'variants.*.barcode.string' => 'Barcode must be valid text.',
        'variants.*.barcode.max' => 'Barcode cannot exceed 255 characters.',
        'variants.*.barcode.regex' => 'Barcode can only contain letters, numbers, hyphens, and underscores.',
        'variants.*.barcode.unique' => 'This barcode is already in use by another variant.',
    ];

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedCategoryFilter()
    {
        $this->resetPage();
    }

    public function updatedItemTypeFilter()
    {
        $this->resetPage();
    }

    public function updatedItemType()
    {
        if ($this->item_type === 'regular') {
            $this->variants = [];
            $this->storage_conditions = null;
            $this->hazard_information = null;
        } elseif ($this->item_type === 'chemical' && empty($this->variants)) {
            $this->addVariant();
        }
    }

    public function toggleShowAddNewItemModal()
    {
        $this->showAddNewItemModal = true;
    }

    public function closeAddNewItemModal()
    {
        $this->showAddNewItemModal = false;
        $this->resetForm();
    }

    public function addVariant()
    {
        $this->variants[] = [
            'volume_per_unit' => '',
            'unit_type' => 'ml',
            'barcode' => ''
        ];
    }

    public function validateVariantUnits()
    {
        $liquidUnits = ['ml', 'L'];
        $solidUnits = ['mg', 'g', 'kg'];
        
        $hasLiquid = false;
        $hasSolid = false;
        
        foreach ($this->variants as $variant) {
            if (!empty($variant['unit_type'])) {
                if (in_array($variant['unit_type'], $liquidUnits)) {
                    $hasLiquid = true;
                } elseif (in_array($variant['unit_type'], $solidUnits)) {
                    $hasSolid = true;
                }
            }
        }
        
        // Warning if mixing liquid and solid units (not an error, just a warning)
        if ($hasLiquid && $hasSolid) {
            // Toast notification for warning
            $this->dispatch('toastMagic', [
                'status' => 'warning',
                'title' => 'Warning!',
                'message' => 'You are mixing liquid and solid units. Ensure this is intentional for this chemical.'
            ]);
        }
    }

    public function getUnitTypeOptions()
    {
        return [
            'ml' => 'Milliliters (ml)',
            'L' => 'Liters (L)', 
            'mg' => 'Milligrams (mg)',
            'g' => 'Grams (g)',
            'kg' => 'Kilograms (kg)'
        ];
    }

    public function duplicateVariant($index)
    {
        if (isset($this->variants[$index])) {
            $originalVariant = $this->variants[$index];
            $duplicatedVariant = [
                'volume_per_unit' => $originalVariant['volume_per_unit'],
                'unit_type' => $originalVariant['unit_type'],
                'barcode' => '' // Clear barcode for duplicate to avoid conflicts
            ];
            
            // Insert after the original variant
            array_splice($this->variants, $index + 1, 0, [$duplicatedVariant]);
            
            // Toast notification for success
            $this->dispatch('toastMagic', [
                'status' => 'success',
                'title' => 'Success!',
                'message' => 'Variant duplicated successfully. Please update the barcode if needed.'
            ]);
        }
    }

    public function validateVariantCombination($index)
    {
        if (!isset($this->variants[$index])) {
            return;
        }
        
        $currentVariant = $this->variants[$index];
        
        // Check for duplicate volume/unit combinations
        foreach ($this->variants as $i => $variant) {
            if ($i !== $index && 
                $variant['volume_per_unit'] == $currentVariant['volume_per_unit'] && 
                $variant['unit_type'] == $currentVariant['unit_type']) {
                // Toast notification for warning
                $this->dispatch('toastMagic', [
                    'status' => 'warning',
                    'title' => 'Warning!',
                    'message' => 'Duplicate volume/unit combination detected. Consider using different values or barcodes to distinguish variants.'
                ]);
                break;
            }
        }
    }

    public function updatedVariants($value, $key)
    {
        // Extract the index from the key (e.g., "0.volume_per_unit" -> 0)
        $index = explode('.', $key)[0];
        
        if (is_numeric($index)) {
            $this->validateVariantCombination($index);
        }
    }

    public function removeVariant($index)
    {
        // Check if this is an existing variant with dependencies
        if (isset($this->variants[$index]['id'])) {
            $variantId = $this->variants[$index]['id'];
            $variant = ItemVariant::find($variantId);
            
            if ($variant) {
                // Check for stock batches
                $stockBatchCount = $variant->stockBatches()->count();
                if ($stockBatchCount > 0) {
                    // Toast notification for error
                    $this->dispatch('toastMagic', [
                        'status' => 'error',
                        'title' => 'Cannot Delete!',
                        'message' => 'Cannot delete variant. It has ' . $stockBatchCount . ' associated stock batch(es).'
                    ]);
                    return;
                }
                
                // Check for transactions
                $transactionCount = $variant->transactions()->count();
                if ($transactionCount > 0) {
                    // Toast notification for error
                    $this->dispatch('toastMagic', [
                        'status' => 'error',
                        'title' => 'Cannot Delete!',
                        'message' => 'Cannot delete variant. It has ' . $transactionCount . ' associated transaction(s).'
                    ]);
                    return;
                }
            }
        }
        
        unset($this->variants[$index]);
        $this->variants = array_values($this->variants);
        
        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => 'Variant removed successfully.'
        ]);
    }

    public function loadItemData($itemId)
    {
        $this->itemId = $itemId;
        $this->isEditing = true;
        $item = InventoryItem::with('variants')->findOrFail($itemId);

        $this->name = $item->name;
        $this->description = $item->description;
        $this->category_id = $item->category_id;
        $this->item_type = $item->item_type;
        $this->min_stock_level = $item->min_stock_level;
        $this->storage_conditions = $item->storage_conditions;
        $this->hazard_information = $item->hazard_information;

        // Load variants for chemical items
        if ($item->item_type === 'chemical') {
            $this->variants = $item->variants->map(function ($variant) {
                return [
                    'id' => $variant->id,
                    'volume_per_unit' => $variant->volume_per_unit,
                    'unit_type' => $variant->unit_type,
                    'barcode' => $variant->barcode
                ];
            })->toArray();
        }

        $this->showAddNewItemModal = true;
    }

    public function saveItem()
    {
        // Validate unit combinations for chemical items
        if ($this->item_type === 'chemical' && !empty($this->variants)) {
            $this->validateVariantUnits();
        }
        
        $this->validate();

        if ($this->isEditing) {
            $this->updateItem();
        } else {
            $this->createItem();
        }
    }

    public function createItem()
    {
        $item = new InventoryItem();
        $item->name = $this->name;
        $item->description = $this->description;
        $item->category_id = $this->category_id;
        $item->item_type = $this->item_type;
        $item->min_stock_level = $this->min_stock_level;
        
        if ($this->item_type === 'chemical') {
            $item->storage_conditions = $this->storage_conditions;
            $item->hazard_information = $this->hazard_information;
        }
        
        $item->save();

        // Create variants for chemical items
        if ($this->item_type === 'chemical' && !empty($this->variants)) {
            foreach ($this->variants as $variantData) {
                if (!empty($variantData['volume_per_unit'])) {
                    ItemVariant::create([
                        'inventory_item_id' => $item->id,
                        'volume_per_unit' => $variantData['volume_per_unit'],
                        'unit_type' => $variantData['unit_type'],
                        'barcode' => $variantData['barcode']
                    ]);
                }
            }
        }

        $this->resetForm();
        $this->closeAddNewItemModal();
        
        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => 'Item created successfully!'
        ]);

        $this->dispatch('itemCreated');
    }

    public function updateItem()
    {
        $item = InventoryItem::findOrFail($this->itemId);
        $item->name = $this->name;
        $item->description = $this->description;
        $item->category_id = $this->category_id;
        $item->item_type = $this->item_type;
        $item->min_stock_level = $this->min_stock_level;
        
        if ($this->item_type === 'chemical') {
            $item->storage_conditions = $this->storage_conditions;
            $item->hazard_information = $this->hazard_information;
        } else {
            $item->storage_conditions = null;
            $item->hazard_information = null;
        }
        
        $item->save();

        // Update variants for chemical items
        if ($this->item_type === 'chemical') {
            // Delete existing variants that are not in the current variants array
            $existingVariantIds = collect($this->variants)->pluck('id')->filter();
            $item->variants()->whereNotIn('id', $existingVariantIds)->delete();

            // Update or create variants
            foreach ($this->variants as $variantData) {
                if (!empty($variantData['volume_per_unit'])) {
                    if (isset($variantData['id'])) {
                        // Update existing variant
                        ItemVariant::where('id', $variantData['id'])->update([
                            'volume_per_unit' => $variantData['volume_per_unit'],
                            'unit_type' => $variantData['unit_type'],
                            'barcode' => $variantData['barcode']
                        ]);
                    } else {
                        // Create new variant
                        ItemVariant::create([
                            'inventory_item_id' => $item->id,
                            'volume_per_unit' => $variantData['volume_per_unit'],
                            'unit_type' => $variantData['unit_type'],
                            'barcode' => $variantData['barcode']
                        ]);
                    }
                }
            }
        } else {
            // Remove all variants if item type changed to regular
            $item->variants()->delete();
        }

        $this->resetForm();
        $this->closeAddNewItemModal();
        
        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => 'Item updated successfully!'
        ]);

        $this->dispatch('itemUpdated');
    }

    public function deleteItem($itemId)
    {
        try {
            $item = InventoryItem::findOrFail($itemId);
            
            // Enhanced dependency checking
            $dependencyCheck = $this->checkItemDependencies($item);
            if (!$dependencyCheck['canDelete']) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Cannot Delete!',
                    'message' => $dependencyCheck['message']
                ]);
                return;
            }

            // Delete variants first (they should have no dependencies at this point)
            $item->variants()->delete();
            
            $item->delete();
            
            // Toast notification for success
            $this->dispatch('toastMagic', [
                'status' => 'success',
                'title' => 'Success!',
                'message' => 'Item deleted successfully!'
            ]);
            
            $this->dispatch('itemDeleted');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'Item not found.'
            ]);
        } catch (\Exception $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'An error occurred while deleting the item. Please try again.'
            ]);
            \Log::error('Item deletion error: ' . $e->getMessage());
        }
    }

    /**
     * Check if item can be deleted by examining dependencies
     */
    private function checkItemDependencies(InventoryItem $item): array
    {
        $stockBatchCount = $item->stockBatches()->count();
        $transactionCount = $item->transactions()->count();
        $alertCount = $item->alerts()->count();
        
        $issues = [];
        
        if ($stockBatchCount > 0) {
            $issues[] = "{$stockBatchCount} stock batch(es)";
        }
        
        if ($transactionCount > 0) {
            $issues[] = "{$transactionCount} transaction(s)";
        }
        
        if ($alertCount > 0) {
            $issues[] = "{$alertCount} alert(s)";
        }
        
        if (!empty($issues)) {
            $message = "Cannot delete item '{$item->name}'. It has " . implode(', ', $issues) . '. ';
            $message .= 'Please resolve these dependencies first or contact an administrator.';
            
            return [
                'canDelete' => false,
                'message' => $message
            ];
        }

        return [
            'canDelete' => true,
            'message' => ''
        ];
    }

    /**
     * Real-time validation methods
     */
    public function updatedName()
    {
        $this->validateOnly('name');
    }

    public function updatedDescription()
    {
        $this->validateOnly('description');
    }

    public function updatedCategoryId()
    {
        $this->validateOnly('category_id');
    }

    public function updatedMinStockLevel()
    {
        $this->validateOnly('min_stock_level');
    }

    public function updatedStorageConditions()
    {
        $this->validateOnly('storage_conditions');
    }

    public function updatedHazardInformation()
    {
        $this->validateOnly('hazard_information');
    }

    /**
     * Enhanced unit conversion validation
     */
    public function validateUnitConversion($fromUnit, $toUnit)
    {
        try {
            $testValue = 1.0;
            \App\Helpers\UnitConverter::convert($testValue, $fromUnit, $toUnit);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Validate variant data comprehensively
     */
    public function validateVariantData($index)
    {
        if (!isset($this->variants[$index])) {
            return;
        }

        $variant = $this->variants[$index];
        
        // Validate volume per unit
        if (!empty($variant['volume_per_unit'])) {
            if (!is_numeric($variant['volume_per_unit']) || $variant['volume_per_unit'] <= 0) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Validation Error!',
                    'message' => 'Volume per unit must be a positive number.'
                ]);
                return;
            }
        }

        // Validate unit type
        if (!empty($variant['unit_type'])) {
            if (!\App\Helpers\UnitConverter::isValidUnit($variant['unit_type'])) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Validation Error!',
                    'message' => 'Invalid unit type selected.'
                ]);
                return;
            }
        }

        // Validate barcode format
        if (!empty($variant['barcode'])) {
            if (!preg_match('/^[a-zA-Z0-9\-_]+$/', $variant['barcode'])) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Validation Error!',
                    'message' => 'Barcode can only contain letters, numbers, hyphens, and underscores.'
                ]);
                return;
            }
        }

        // Check for duplicate combinations
        $this->validateVariantCombination($index);
    }

    public function exportItems()
    {
        $query = InventoryItem::query()
            ->with(['category', 'variants'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->categoryFilter, function($query) {
                $query->where('category_id', $this->categoryFilter);
            })
            ->when($this->itemTypeFilter, function($query) {
                $query->where('item_type', $this->itemTypeFilter);
            });

        return Excel::download(new ItemExport($query), 'items.xlsx');
    }

    public function exportItemsPdf()
    {
        $data = InventoryItem::query()
            ->with(['category', 'variants'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->categoryFilter, function($query) {
                $query->where('category_id', $this->categoryFilter);
            })
            ->when($this->itemTypeFilter, function($query) {
                $query->where('item_type', $this->itemTypeFilter);
            })
            ->orderBy('name')
            ->get();

        $pdf = Pdf::loadView('exports.item-pdf', [
            'data' => $data,
            'title' => 'Items Report',
            'generatedAt' => now()->format('Y-m-d H:i:s')
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'items_report_' . now()->format('Y-m-d_H-i-s') . '.pdf');
    }

    public function resetForm()
    {
        $this->reset([
            'name',
            'description',
            'category_id',
            'item_type',
            'min_stock_level',
            'storage_conditions',
            'hazard_information',
            'variants',
            'isEditing',
            'itemId'
        ]);
        
        $this->item_type = 'regular';
    }

    public function render()
    {
        $items = InventoryItem::query()
            ->with(['category', 'variants'])
            ->withCount('stockBatches')
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->categoryFilter, function($query) {
                $query->where('category_id', $this->categoryFilter);
            })
            ->when($this->itemTypeFilter, function($query) {
                $query->where('item_type', $this->itemTypeFilter);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        $categories = Category::orderBy('name')->get();

        return view('livewire.inventory.item-list', [
            'items' => $items,
            'categories' => $categories
        ]);
    }
}