<?php

namespace App\Livewire\Inventory;

use App\Models\StockAlert;
use App\Models\InventoryItem;
use App\Models\StockBatch;
use App\Models\StockTransaction;
use Livewire\Component;
use Carbon\Carbon;

class Dashboard extends Component
{
    public function mount()
    {
        // Generate alerts when dashboard loads
        $this->generateAlerts();
    }

    private function generateAlerts()
    {
        // Generate low stock alerts
        $this->generateLowStockAlerts();
        
        // Generate expiry alerts
        $this->generateExpiryAlerts();
    }

    private function generateLowStockAlerts()
    {
        $items = InventoryItem::whereNotNull('min_stock_level')
            ->where('min_stock_level', '>', 0)
            ->get();

        foreach ($items as $item) {
            $currentStock = $item->getCurrentStock();
            
            if ($currentStock <= $item->min_stock_level) {
                // Check if alert already exists and is unresolved
                $existingAlert = StockAlert::where('inventory_item_id', $item->id)
                    ->where('alert_type', 'low_stock')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    StockAlert::create([
                        'inventory_item_id' => $item->id,
                        'alert_type' => 'low_stock',
                        'alert_message' => "Low stock alert: {$item->name} has {$currentStock} units remaining (minimum: {$item->min_stock_level})",
                        'resolved' => false,
                    ]);
                }
            }
        }
    }

    private function generateExpiryAlerts($daysAhead = 30)
    {
        $batches = StockBatch::whereNotNull('expiry_date')
            ->where('current_quantity', '>', 0)
            ->get();

        foreach ($batches as $batch) {
            $daysUntilExpiry = now()->diffInDays($batch->expiry_date, false);
            
            // Create expiry alerts for items expiring within the specified timeframe
            if ($daysUntilExpiry <= $daysAhead && $daysUntilExpiry >= 0) {
                // Check if alert already exists and is unresolved
                $existingAlert = StockAlert::where('batch_id', $batch->id)
                    ->where('alert_type', 'expiry')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    $message = $daysUntilExpiry == 0 
                        ? "Expiry alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expires today!"
                        : "Expiry alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expires in {$daysUntilExpiry} days";

                    StockAlert::create([
                        'inventory_item_id' => $batch->inventory_item_id,
                        'batch_id' => $batch->id,
                        'alert_type' => 'expiry',
                        'alert_message' => $message,
                        'resolved' => false,
                    ]);
                }
            }
            
            // Create expired alerts for items that have already expired
            if ($daysUntilExpiry < 0) {
                $existingAlert = StockAlert::where('batch_id', $batch->id)
                    ->where('alert_type', 'expiry')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    $daysExpired = abs($daysUntilExpiry);
                    StockAlert::create([
                        'inventory_item_id' => $batch->inventory_item_id,
                        'batch_id' => $batch->id,
                        'alert_type' => 'expiry',
                        'alert_message' => "Expired alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expired {$daysExpired} days ago",
                        'resolved' => false,
                    ]);
                }
            }
        }
    }

    public function render()
    {
        // Get critical alerts (unresolved)
        $criticalAlerts = StockAlert::with(['inventoryItem', 'batch'])
            ->where('resolved', false)
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Get recent transactions
        $recentTransactions = StockTransaction::with(['inventoryItem', 'batch'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Get low stock items
        $lowStockItems = InventoryItem::whereNotNull('min_stock_level')
            ->where('min_stock_level', '>', 0)
            ->get()
            ->filter(function ($item) {
                return $item->getCurrentStock() <= $item->min_stock_level;
            })
            ->take(5);

        // Get expiring batches (next 30 days)
        $expiringBatches = StockBatch::with(['inventoryItem', 'supplier'])
            ->whereNotNull('expiry_date')
            ->where('current_quantity', '>', 0)
            ->where('expiry_date', '>=', now())
            ->where('expiry_date', '<=', now()->addDays(30))
            ->orderBy('expiry_date', 'asc')
            ->limit(5)
            ->get();

        // Get dashboard statistics
        $stats = [
            'total_items' => InventoryItem::count(),
            'total_batches' => StockBatch::where('current_quantity', '>', 0)->count(),
            'critical_alerts' => StockAlert::where('resolved', false)->count(),
            'low_stock_count' => $lowStockItems->count(),
            'expiring_soon' => $expiringBatches->count(),
            'total_value' => StockBatch::where('current_quantity', '>', 0)->sum('unit_cost'),
        ];

        // Get alert breakdown
        $alertBreakdown = [
            'low_stock' => StockAlert::where('alert_type', 'low_stock')->where('resolved', false)->count(),
            'expiry' => StockAlert::where('alert_type', 'expiry')->where('resolved', false)->count(),
        ];

        return view('livewire.inventory.dashboard', [
            'criticalAlerts' => $criticalAlerts,
            'recentTransactions' => $recentTransactions,
            'lowStockItems' => $lowStockItems,
            'expiringBatches' => $expiringBatches,
            'stats' => $stats,
            'alertBreakdown' => $alertBreakdown,
        ]);
    }
}