<?php

namespace App\Livewire\Inventory;

use App\Models\Category;
use App\Exports\CategoryExport;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;

class CategoryList extends Component
{
    use WithPagination;

    public $name;
    public $description;
    public $search = '';
    public $showAddNewCategoryModal = false;
    public $isEditing = false;
    public $categoryId = null;

    protected $listeners = [
        'editCategory' => 'loadCategoryData',
        'resetForm' => 'resetForm'
    ];

    protected function rules()
    {
        return [
            'name' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z0-9\s\-_\.]+$/',
                'unique:categories,name,' . ($this->categoryId ?? 'NULL')
            ],
            'description' => 'nullable|string|max:1000',
        ];
    }

    protected $messages = [
        'name.required' => 'Category name is required.',
        'name.string' => 'Category name must be a valid text.',
        'name.max' => 'Category name cannot exceed 255 characters.',
        'name.regex' => 'Category name can only contain letters, numbers, spaces, hyphens, underscores, and periods.',
        'name.unique' => 'A category with this name already exists.',
        'description.string' => 'Description must be valid text.',
        'description.max' => 'Description cannot exceed 1000 characters.',
    ];

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function toggleShowAddNewCategoryModal()
    {
        $this->showAddNewCategoryModal = true;
    }

    public function closeAddNewCategoryModal()
    {
        $this->showAddNewCategoryModal = false;
        $this->resetForm();
    }

    public function loadCategoryData($categoryId)
    {
        $this->categoryId = $categoryId;
        $this->isEditing = true;
        $category = Category::findOrFail($categoryId);

        $this->name = $category->name;
        $this->description = $category->description;
        $this->showAddNewCategoryModal = true;
    }

    public function saveCategory()
    {
        $this->validate();

        if ($this->isEditing) {
            $this->updateCategory();
        } else {
            $this->createCategory();
        }
    }

    public function createCategory()
    {
        $category = new Category();
        $category->name = $this->name;
        $category->description = $this->description;
        $category->save();

        $this->resetForm();
        $this->closeAddNewCategoryModal();
        
        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => 'Category created successfully!'
        ]);

        $this->dispatch('categoryCreated');
    }

    public function updateCategory()
    {
        $category = Category::findOrFail($this->categoryId);
        $category->name = $this->name;
        $category->description = $this->description;
        $category->save();

        $this->resetForm();
        $this->closeAddNewCategoryModal();
        
        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => 'Category updated successfully!'
        ]);

        $this->dispatch('categoryUpdated');
    }

    public function deleteCategory($categoryId)
    {
        try {
            $category = Category::findOrFail($categoryId);
            
            // Enhanced dependency checking
            $dependencyCheck = $this->checkCategoryDependencies($category);
            if (!$dependencyCheck['canDelete']) {
                // Toast notification for error
                $this->dispatch('toastMagic', [
                    'status' => 'error',
                    'title' => 'Cannot Delete!',
                    'message' => $dependencyCheck['message']
                ]);
                return;
            }

            $category->delete();
            
            // Toast notification for success
            $this->dispatch('toastMagic', [
                'status' => 'success',
                'title' => 'Success!',
                'message' => 'Category deleted successfully!'
            ]);
            
            $this->dispatch('categoryDeleted');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'Category not found.'
            ]);
        } catch (\Exception $e) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'An error occurred while deleting the category. Please try again.'
            ]);
            \Log::error('Category deletion error: ' . $e->getMessage());
        }
    }

    /**
     * Check if category can be deleted by examining dependencies
     */
    private function checkCategoryDependencies(Category $category): array
    {
        $itemCount = $category->inventoryItems()->count();
        
        if ($itemCount > 0) {
            $itemNames = $category->inventoryItems()
                ->limit(3)
                ->pluck('name')
                ->toArray();
            
            $message = "Cannot delete category '{$category->name}'. It has {$itemCount} associated item(s)";
            if ($itemCount <= 3) {
                $message .= ': ' . implode(', ', $itemNames);
            } else {
                $message .= ' including: ' . implode(', ', $itemNames) . ' and ' . ($itemCount - 3) . ' more';
            }
            $message .= '. Please remove or reassign these items first.';
            
            return [
                'canDelete' => false,
                'message' => $message
            ];
        }

        return [
            'canDelete' => true,
            'message' => ''
        ];
    }

    /**
     * Validate category name in real-time
     */
    public function updatedName()
    {
        $this->validateOnly('name');
    }

    /**
     * Validate description in real-time
     */
    public function updatedDescription()
    {
        $this->validateOnly('description');
    }

    public function exportCategories()
    {
        $query = Category::query();
        
        if (!empty($this->search)) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%');
            });
        }

        return Excel::download(new CategoryExport($query), 'categories.xlsx');
    }

    public function resetForm()
    {
        $this->reset([
            'name',
            'description',
            'isEditing',
            'categoryId'
        ]);
    }

    public function render()
    {
        $categories = Category::query()
            ->withCount('inventoryItems')
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('livewire.inventory.category-list', [
            'categories' => $categories
        ]);
    }
}