<?php

namespace App\Livewire\Inventory;

use App\Models\StockAlert;
use App\Models\InventoryItem;
use App\Models\StockBatch;
use App\Exports\StockAlertExport;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class AlertManagement extends Component
{
    use WithPagination;

    public $search = '';
    public $alertTypeFilter = '';
    public $statusFilter = '';
    public $itemFilter = '';
    
    public $showResolveModal = false;
    public $selectedAlert = null;
    public $resolveNotes = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'alertTypeFilter' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'itemFilter' => ['except' => ''],
    ];

    public function mount()
    {
        // Generate alerts when the page loads
        $this->generateAlerts();
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingAlertTypeFilter()
    {
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }

    public function updatingItemFilter()
    {
        $this->resetPage();
    }

    public function generateAlerts()
    {
        // Generate low stock alerts
        $this->generateLowStockAlerts();
        
        // Generate expiry alerts
        $this->generateExpiryAlerts();
        
        // Toast notification for info
        $this->dispatch('toastMagic', [
            'status' => 'info',
            'title' => 'Info',
            'message' => 'Alerts have been refreshed.'
        ]);
    }

    private function generateLowStockAlerts()
    {
        $items = InventoryItem::whereNotNull('min_stock_level')
            ->where('min_stock_level', '>', 0)
            ->get();

        foreach ($items as $item) {
            $currentStock = $item->getCurrentStock();
            
            if ($currentStock <= $item->min_stock_level) {
                // Check if alert already exists and is unresolved
                $existingAlert = StockAlert::where('inventory_item_id', $item->id)
                    ->where('alert_type', 'low_stock')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    StockAlert::create([
                        'inventory_item_id' => $item->id,
                        'alert_type' => 'low_stock',
                        'alert_message' => "Low stock alert: {$item->name} has {$currentStock} units remaining (minimum: {$item->min_stock_level})",
                        'resolved' => false,
                    ]);
                }
            }
        }
    }

    private function generateExpiryAlerts($daysAhead = 30)
    {
        $batches = StockBatch::whereNotNull('expiry_date')
            ->where('current_quantity', '>', 0)
            ->get();

        foreach ($batches as $batch) {
            $daysUntilExpiry = now()->diffInDays($batch->expiry_date, false);
            
            // Create expiry alerts for items expiring within the specified timeframe
            if ($daysUntilExpiry <= $daysAhead && $daysUntilExpiry >= 0) {
                // Check if alert already exists and is unresolved
                $existingAlert = StockAlert::where('batch_id', $batch->id)
                    ->where('alert_type', 'expiry')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    $message = $daysUntilExpiry == 0 
                        ? "Expiry alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expires today!"
                        : "Expiry alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expires in {$daysUntilExpiry} days";

                    StockAlert::create([
                        'inventory_item_id' => $batch->inventory_item_id,
                        'batch_id' => $batch->id,
                        'alert_type' => 'expiry',
                        'alert_message' => $message,
                        'resolved' => false,
                    ]);
                }
            }
            
            // Create expired alerts for items that have already expired
            if ($daysUntilExpiry < 0) {
                $existingAlert = StockAlert::where('batch_id', $batch->id)
                    ->where('alert_type', 'expiry')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    $daysExpired = abs($daysUntilExpiry);
                    StockAlert::create([
                        'inventory_item_id' => $batch->inventory_item_id,
                        'batch_id' => $batch->id,
                        'alert_type' => 'expiry',
                        'alert_message' => "Expired alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expired {$daysExpired} days ago",
                        'resolved' => false,
                    ]);
                }
            }
        }
    }

    public function openResolveModal($alertId)
    {
        $this->selectedAlert = StockAlert::findOrFail($alertId);
        $this->resolveNotes = '';
        $this->showResolveModal = true;
    }

    public function closeResolveModal()
    {
        $this->showResolveModal = false;
        $this->selectedAlert = null;
        $this->resolveNotes = '';
    }

    public function resolveAlert()
    {
        if (!$this->selectedAlert) {
            return;
        }

        $this->selectedAlert->resolve(Auth::user()->name ?? 'System');
        
        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => 'Alert has been resolved successfully.'
        ]);
        
        $this->closeResolveModal();
    }

    public function bulkResolve($alertIds)
    {
        if (empty($alertIds)) {
            // Toast notification for error
            $this->dispatch('toastMagic', [
                'status' => 'error',
                'title' => 'Error!',
                'message' => 'No alerts selected.'
            ]);
            return;
        }

        $resolvedCount = StockAlert::whereIn('id', $alertIds)
            ->where('resolved', false)
            ->update([
                'resolved' => true,
                'resolved_by' => Auth::user()->name ?? 'System',
                'resolved_at' => now(),
            ]);

        // Toast notification for success
        $this->dispatch('toastMagic', [
            'status' => 'success',
            'title' => 'Success!',
            'message' => "{$resolvedCount} alerts have been resolved."
        ]);
    }

    public function exportAlerts()
    {
        $query = StockAlert::query()
            ->with(['inventoryItem.category', 'batch'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('alert_message', 'like', '%' . $this->search . '%')
                      ->orWhereHas('inventoryItem', function($itemQuery) {
                          $itemQuery->where('name', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('batch', function($batchQuery) {
                          $batchQuery->where('batch_number', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->alertTypeFilter, function($query) {
                $query->where('alert_type', $this->alertTypeFilter);
            })
            ->when($this->statusFilter === 'resolved', function($query) {
                $query->where('resolved', true);
            })
            ->when($this->statusFilter === 'unresolved', function($query) {
                $query->where('resolved', false);
            })
            ->when($this->itemFilter, function($query) {
                $query->where('inventory_item_id', $this->itemFilter);
            });

        return Excel::download(new StockAlertExport($query), 'stock_alerts.xlsx');
    }

    public function exportAlertsPdf()
    {
        $data = StockAlert::query()
            ->with(['inventoryItem.category', 'batch'])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('alert_message', 'like', '%' . $this->search . '%')
                      ->orWhereHas('inventoryItem', function($itemQuery) {
                          $itemQuery->where('name', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('batch', function($batchQuery) {
                          $batchQuery->where('batch_number', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->alertTypeFilter, function($query) {
                $query->where('alert_type', $this->alertTypeFilter);
            })
            ->when($this->statusFilter === 'resolved', function($query) {
                $query->where('resolved', true);
            })
            ->when($this->statusFilter === 'unresolved', function($query) {
                $query->where('resolved', false);
            })
            ->when($this->itemFilter, function($query) {
                $query->where('inventory_item_id', $this->itemFilter);
            })
            ->orderBy('resolved', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        $pdf = Pdf::loadView('exports.stock-alert-pdf', [
            'data' => $data,
            'title' => 'Stock Alerts Report',
            'generatedAt' => now()->format('Y-m-d H:i:s')
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'stock_alerts_report_' . now()->format('Y-m-d_H-i-s') . '.pdf');
    }

    public function render()
    {
        $query = StockAlert::with(['inventoryItem', 'batch.supplier'])
            ->orderBy('resolved', 'asc')
            ->orderBy('created_at', 'desc');

        // Apply search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('alert_message', 'like', '%' . $this->search . '%')
                  ->orWhereHas('inventoryItem', function($itemQuery) {
                      $itemQuery->where('name', 'like', '%' . $this->search . '%');
                  })
                  ->orWhereHas('batch', function($batchQuery) {
                      $batchQuery->where('batch_number', 'like', '%' . $this->search . '%');
                  });
            });
        }

        // Apply alert type filter
        if ($this->alertTypeFilter) {
            $query->where('alert_type', $this->alertTypeFilter);
        }

        // Apply status filter
        if ($this->statusFilter === 'resolved') {
            $query->where('resolved', true);
        } elseif ($this->statusFilter === 'unresolved') {
            $query->where('resolved', false);
        }

        // Apply item filter
        if ($this->itemFilter) {
            $query->where('inventory_item_id', $this->itemFilter);
        }

        $alerts = $query->paginate(15);

        // Get filter options
        $inventoryItems = InventoryItem::orderBy('name')->get();
        
        // Get alert statistics
        $alertStats = [
            'total' => StockAlert::count(),
            'unresolved' => StockAlert::where('resolved', false)->count(),
            'low_stock' => StockAlert::where('alert_type', 'low_stock')->where('resolved', false)->count(),
            'expiry' => StockAlert::where('alert_type', 'expiry')->where('resolved', false)->count(),
        ];

        return view('livewire.inventory.alert-management', [
            'alerts' => $alerts,
            'inventoryItems' => $inventoryItems,
            'alertStats' => $alertStats,
        ]);
    }
}