<?php

namespace App\Livewire\Instruments;

use App\Mail\QuotationMail;
use App\Models\Instrument;
use Illuminate\Support\Facades\Mail;
use Livewire\Component;
use Livewire\WithPagination;

class PublicInstrumentsList extends Component
{
    use WithPagination;

    public $selectedSector = 'internal';
    public $selectAll = false;
    public $selected = [];
    public $quantities = [];
    public $showQuotationModal = false;

    protected $messages = [
        'quantities.*.sample.max' => 'Sample quantity cannot exceed 20.',
        'quantities.*.slot.max' => 'Slot quantity cannot exceed 20.',
        'quantities.*.booking.max' => 'Booking quantity cannot exceed 20.'
    ];

    public function updatedQuantities($value, $key)
    {
        $parts = explode('.', $key);
        if (count($parts) === 3 && in_array($parts[2], ['sample', 'slot', 'booking']) && $value > 20) {
            $this->quantities[$parts[0]][$parts[2]] = 20;
        }
    }

    protected $paginationTheme = 'bootstrap';

    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->selected = $this->instruments->pluck('id')->map(fn($id) => (string) $id)->toArray();
        } else {
            $this->selected = [];
        }
    }

    public function updateValueInCalculator($instrumentId)
    {
        if (in_array($instrumentId, $this->selected)) {
            $this->selected = array_diff($this->selected, [$instrumentId]);
        } else {
            $this->selected[] = $instrumentId;
        }
    }

    public function getInstrumentsProperty()
    {
        return Instrument::where('public_visibility_status', 1)
            ->where('operating_status', 'working')
            ->paginate(10);
    }

    public $quotationForm = [
        'customerName' => '',
        'organization' => '',
        'email' => '',
        'phone' => ''
    ];

    protected $rules = [
        'quotationForm.customerName' => 'required|string|max:255',
        'quotationForm.organization' => 'required|string|max:255',
        'quotationForm.email' => 'required|email|max:255',
        'quotationForm.phone' => 'required|string|max:20'
    ];

    public function openQuotationModal()
    {
        if (empty($this->selected)) {
            session()->flash('error', 'Please select at least one instrument to request a quotation.');
            return;
        }

        $this->showQuotationModal = true;
    }

    public function closeQuotationModal()
    {
        $this->showQuotationModal = false;
        $this->reset('quotationForm');
    }

    public function sendQuotation()
    {
        $this->validate();

        $selectedInstruments = [];
        $charges = $this->calculatedCharges;

        foreach ($this->selected as $instrumentId) {
            $instrument = $this->instruments->firstWhere('id', $instrumentId);
            if (!$instrument || !isset($this->quantities[$instrumentId])) continue;

            $bookingType = $instrument->booking_type;
            if ($bookingType === 'both') {
                $bookingType = $this->quantities[$instrumentId]['type'] ?? '';
            }

            $instrumentData = [
                'name' => $instrument->name,
                'type' => $bookingType,
                'quantity' => 0,
                'rate' => 0,
                'gst' => 0,
                'amount' => 0
            ];

            if ($bookingType === 'sample' && isset($this->quantities[$instrumentId]['sample'])) {
                $qty = intval($this->quantities[$instrumentId]['sample']);
                $rate = floatval($instrument->booking_cost[$this->selectedSector]['sample']);
                $gst = floatval($instrument->booking_cost[$this->selectedSector]['sample_gst']);
                $amount = $qty * $rate;
                $instrumentData['quantity'] = $qty;
                $instrumentData['rate'] = $rate;
                $instrumentData['gst'] = $gst;
                $instrumentData['amount'] = $amount + ($amount * $gst / 100);
            } elseif ($bookingType === 'slot') {
                if (isset($this->quantities[$instrumentId]['slot'])) {
                    $qty = intval($this->quantities[$instrumentId]['slot']);
                    $rate = floatval($instrument->booking_cost[$this->selectedSector]['slot']);
                    $gst = floatval($instrument->booking_cost[$this->selectedSector]['slot_gst']);
                    $amount = $qty * $rate;
                    $instrumentData['quantity'] = $qty;
                    $instrumentData['rate'] = $rate;
                    $instrumentData['gst'] = $gst;
                    $instrumentData['amount'] = $amount + ($amount * $gst / 100);
                }
                if (isset($this->quantities[$instrumentId]['booking'])) {
                    $qty = intval($this->quantities[$instrumentId]['booking']);
                    $rate = floatval($instrument->booking_cost[$this->selectedSector]['booking']);
                    $gst = floatval($instrument->booking_cost[$this->selectedSector]['booking_gst']);
                    $amount = $qty * $rate;
                    $instrumentData['quantity'] = $qty;
                    $instrumentData['rate'] = $rate;
                    $instrumentData['gst'] = $gst;
                    $instrumentData['amount'] = $amount + ($amount * $gst / 100);
                }
            }

            $selectedInstruments[] = $instrumentData;
        }

//        Mail::to($this->quotationForm['email'])
//            ->send(new QuotationMail(
//                $this->quotationForm['customerName'],
//                $this->quotationForm['organization'],
//                $this->quotationForm['email'],
//                $this->quotationForm['phone'],
//                $selectedInstruments,
//                $charges
//            ));

        Mail::to($this->quotationForm['email'])
            ->queue(new QuotationMail(
                $this->quotationForm['customerName'],
                $this->quotationForm['organization'],
                $this->quotationForm['email'],
                $this->quotationForm['phone'],
                $selectedInstruments,
                $charges
            ));

        $this->showQuotationModal = false;
        $this->reset('quotationForm');
        session()->flash('success', 'Quotation has been sent to your email.');
    }

    public function getCalculatedChargesProperty()
    {
        $instrumentCharges = 0;
        $gstCharges = 0;

        foreach ($this->selected as $instrumentId) {
            $instrument = $this->instruments->firstWhere('id', $instrumentId);
            if (!$instrument || !isset($this->quantities[$instrumentId])) continue;

            $bookingType = $instrument->booking_type;
            if ($bookingType === 'both') {
                $bookingType = $this->quantities[$instrumentId]['type'] ?? '';
            }

            if ($bookingType === 'sample') {
                $sampleQty = intval($this->quantities[$instrumentId]['sample'] ?? 0);
                if ($sampleQty > 0 && isset($instrument->booking_cost[$this->selectedSector]['sample'])) {
                    $sampleRate = floatval($instrument->booking_cost[$this->selectedSector]['sample']);
                    $sampleGst = floatval($instrument->booking_cost[$this->selectedSector]['sample_gst']);

                    $charge = $sampleQty * $sampleRate;
                    $instrumentCharges += $charge;
                    $gstCharges += ($charge * $sampleGst / 100);
                }
            } elseif ($bookingType === 'slot') {
                $slotQty = intval($this->quantities[$instrumentId]['slot'] ?? 0);
                $bookingQty = intval($this->quantities[$instrumentId]['booking'] ?? 0);

                if ($slotQty > 0 && isset($instrument->booking_cost[$this->selectedSector]['slot'])) {
                    $slotRate = floatval($instrument->booking_cost[$this->selectedSector]['slot']);
                    $slotGst = floatval($instrument->booking_cost[$this->selectedSector]['slot_gst']);
                    $charge = $slotQty * $slotRate;
                    $instrumentCharges += $charge;
                    $gstCharges += ($charge * $slotGst / 100);
                }

                if ($bookingQty > 0 && isset($instrument->booking_cost[$this->selectedSector]['booking'])) {
                    $bookingRate = floatval($instrument->booking_cost[$this->selectedSector]['booking']);
                    $bookingGst = floatval($instrument->booking_cost[$this->selectedSector]['booking_gst']);
                    $charge = $bookingQty * $bookingRate;
                    $instrumentCharges += $charge;
                    $gstCharges += ($charge * $bookingGst / 100);
                }
            }
        }

        return [
            'instrument' => round($instrumentCharges, 2),
            'gst' => round($gstCharges, 2),
            'total' => round($instrumentCharges + $gstCharges, 2)
        ];
    }

    public function render()
    {
        $charges = $this->calculatedCharges;
        return view('livewire.instruments.public-instruments-list', [
            'instruments' => $this->instruments,
            'charges' => $charges
        ]);
    }
}


