<?php

namespace App\Livewire\Instruments;

use App\Models\OtherComplaint;
use App\Models\Student;
use App\Models\Instrument;
use Livewire\Component;
use Livewire\WithPagination;

class OtherComplaints extends Component
{
    use WithPagination;

    public $search = '';
    public $studentSearch = '';
    public $instrumentSearch = '';
    public $status = 'All';

    // Status Update Modal Properties
    public $showStatusModal = false;
    public $selectedComplaint = null;
    public $complaintStatus = '';
    public $complaintRemark = '';

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStudentSearch()
    {
        $this->resetPage();
    }

    public function updatingInstrumentSearch()
    {
        $this->resetPage();
    }

    public function updatingStatus()
    {
        $this->resetPage();
    }

    public function updateComplaintStatus($complaintId)
    {
        $this->selectedComplaint = OtherComplaint::findOrFail($complaintId);
        $this->complaintStatus = $this->selectedComplaint->status;
        $this->complaintRemark = $this->selectedComplaint->remark ?? '';
        $this->showStatusModal = true;
    }

    public function saveComplaintStatus()
    {
        $this->validate([
            'complaintStatus' => 'required|in:pending,resolved,work_in_progress',
            'complaintRemark' => 'nullable|string|max:500'
        ]);

        $this->selectedComplaint->update([
            'status' => $this->complaintStatus,
            'remark' => $this->complaintRemark,
            'resolved_by' => auth()->user()->name
        ]);

        session()->flash('success', 'Complaint status updated successfully!');
        $this->showStatusModal = false;
        $this->reset(['selectedComplaint', 'complaintStatus', 'complaintRemark']);
    }

    public function render()
    {
        $query = OtherComplaint::with(['instrument', 'student'])->latest();

        // If user is student, show only their complaints
        if (auth()->user()->hasRole('student')) {
            $studentId = auth()->user()->students->first()->id ?? null;
            if ($studentId) {
                $query->where('student_id', $studentId);
            }
        }

        // Apply search filters
        if ($this->search) {
            $query->where(function($q) {
                $q->whereHas('student', function($q) {
                    $q->where('first_name', 'like', '%' . $this->search . '%')
                      ->orWhere('last_name', 'like', '%' . $this->search . '%');
                })
                ->orWhereHas('instrument', function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%');
                })
                ->orWhere('complaint', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->studentSearch) {
            $query->whereHas('student', function($q) {
                $q->where('first_name', 'like', '%' . $this->studentSearch . '%')
                  ->orWhere('last_name', 'like', '%' . $this->studentSearch . '%');
            });
        }

        if ($this->instrumentSearch) {
            $query->whereHas('instrument', function($q) {
                $q->where('name', 'like', '%' . $this->instrumentSearch . '%');
            });
        }

        if ($this->status !== 'All') {
            $query->where('status', $this->status);
        }

        return view('livewire.instruments.other-complaints', [
            'complaints' => $query->paginate(10),
            'students' => Student::orderBy('first_name')->get(),
            'instruments' => Instrument::orderBy('name')->get()
        ]);
    }
}
