<?php

namespace App\Livewire\Instruments;

use App\Models\Booking;
use Livewire\Component;

class InstrumentView extends Component
{
    public $instrument;
    public $showBookingDetailsTable = false;
    public $showServicesTable = false;
    public $showFullDescription = false;
    public $showFullAccessoryDescription = false;

    public function toggleDescription()
    {
        $this->showFullDescription = !$this->showFullDescription;
    }

    public function toggleAccessoryDescription()
    {
        $this->showFullAccessoryDescription = !$this->showFullAccessoryDescription;
    }

    protected $listeners = [
        'hideForm' => 'handleInstrumentBookingTable',
        'hideServiceRecords' => 'toggleServiceRecords',
    ];

    public function handleInstrumentBookingTable()
    {
        $this->showBookingDetailsTable = !$this->showBookingDetailsTable;
    }

    public function showServiceTable()
    {
        $this->showServicesTable = !$this->showServicesTable;
    }

    public function hideViewInstrument()
    {
        $this->dispatch('hideViewInstrument');
    }

    public function showForm()
    {
        $this->showBookingDetailsTable = !$this->showBookingDetailsTable;
    }

    public function toggleServiceRecords()
    {
        $this->showServiceRecords = !$this->showServiceRecords;
        $this->dispatch('serviceRecordsToggled', $this->showServiceRecords);
    }

    public function deleteAccessory($id)
    {
        $accessory = \App\Models\InstrumentAccessorie::find($id);
        if ($accessory) {
            if ($accessory->photo && \Storage::disk('public')->exists($accessory->photo)) {
                \Storage::disk('public')->delete($accessory->photo);
            }
            $accessory->delete();
            $this->instrument->refresh();
            $this->dispatch('toastSuccess', 'Accessory deleted successfully.');
        }
    }

    public function deletePhoto($index)
    {
        $photos = json_decode($this->instrument->photos, true) ?? [];
        if (isset($photos[$index])) {
            $photo = $photos[$index];
            if (\Storage::disk('public')->exists($photo)) {
                \Storage::disk('public')->delete($photo);
            }
            unset($photos[$index]);
            $this->instrument->photos = json_encode(array_values($photos));
            $this->instrument->save();
            $this->dispatch('toastSuccess', 'Photo deleted successfully.');
        }
    }

    public function deleteFile($type)
    {
        if ($this->instrument->$type) {
            if (\Storage::disk('public')->exists($this->instrument->$type)) {
                \Storage::disk('public')->delete($this->instrument->$type);
            }
            $this->instrument->$type = null;
            $this->instrument->save();
            $this->dispatch('toastSuccess', ucfirst(str_replace('_', ' ', $type)) . ' deleted successfully.');
        }
    }

    public function render()
    {
        $bookings = Booking::where('instrument_id', $this->instrument->id)
            ->latest()
            ->take(5)
            ->get();

        // Calculate stats for the instrument
        $totalBookings = $this->instrument->bookings()->count();
        $totalServices = $this->instrument->services()->count();
        
        // Calculate total expense on instrument (from services)
        $totalExpense = $this->instrument->services()
            ->whereNotNull('invoice_details')
            ->get()
            ->sum(function($service) {
                $invoiceDetails = json_decode($service->invoice_details, true);
                return $invoiceDetails['invoice_cost'] ?? 0;
            });

        // Calculate total revenue from bookings
        $totalRevenue = $this->instrument->bookings()
            ->where('status', 'confirmed')
            ->sum('booking_cost');

        // Calculate instrument age
        $manufacturingDate = $this->instrument->purchaseInformation?->manufacturing_date;
        $instrumentAge = $manufacturingDate ? 
            \Carbon\Carbon::parse($manufacturingDate)->diffInYears(now()) . ' years' : 
            'N/A';

        return view('livewire.instruments.instrument-view', [
            'bookings' => $bookings,
            'totalBookings' => $totalBookings,
            'totalServices' => $totalServices,
            'totalExpense' => $totalExpense,
            'totalRevenue' => $totalRevenue,
            'instrumentAge' => $instrumentAge
        ]);
    }
}
