<?php

namespace App\Livewire\Instruments;

use App\Models\InstrumentTerm;
use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\Rule;

class InstrumentTermsForm extends Component
{
    public $instrument;
    public $showMoreBookingFields = false;
    public $isApprovelRequired = false;
    public $users = [];
    public $isEditing = false;
    public $instrumentTerm = null;
    public $editorId;
    public $instrument_sample_questions = [];

    #[Rule('nullable|integer|min:1')]
    public $can_cancel_before_booking_slot;

    #[Rule('nullable|string')]
    public $maximum_sample_booking_per_day;

    #[Rule('nullable|in:day,week,month')]
    public $maximum_booking_type = '';

    #[Rule('nullable|in:pi,student')]
    public $maximum_booking_for;

    #[Rule('nullable|integer|min:1')]
    public $maximum_bookings;

    #[Rule('required|boolean')]
    public $approval_required = false;

    #[Rule('nullable|required_if:approval_required,1')]
    public $approval_from;

    #[Rule('nullable|string')]
    public $sop;

    #[Rule('nullable|integer|min:15|max:480')]
    public $booking_slot_duration = 30;

    public function mount()
    {
        $this->editorId = 'sop-editor-' . uniqid();

        if ($this->instrument->instrumentTerm) {
            $this->isEditing = true;
            $this->instrumentTerm = $this->instrument->instrumentTerm;
            $this->can_cancel_before_booking_slot = $this->instrumentTerm->can_cancel_before_booking_slot;
            $this->maximum_booking_type = $this->instrumentTerm->maximum_booking_type;
            $this->maximum_booking_for = $this->instrumentTerm->maximum_booking_for;
            $this->maximum_bookings = $this->instrumentTerm->maximum_bookings;
            $this->approval_required = $this->instrumentTerm->approval_required;
            $this->approval_from = $this->instrumentTerm->approval_from;
            $this->sop = $this->instrumentTerm->sop;
            $this->maximum_sample_booking_per_day = $this->instrumentTerm->maximum_sample_booking_per_day;
            $this->booking_slot_duration = $this->instrumentTerm->booking_slot_duration ?? 30;
            $this->instrument_sample_questions = $this->instrumentTerm->instrument_sample_questions ? json_decode($this->instrumentTerm->instrument_sample_questions, true) : [];

            if (!empty($this->maximum_booking_type)) {
                $this->showMoreBookingFields = true;
            }

            if ($this->approval_required) {
                $this->isApprovelRequired = true;
                $this->users = User::whereDoesntHave('roles', function($query) {
                    $query->where('name', 'student');
                })->get(['id', 'name', 'email']);
            }
        }
    }

    public function updatedMaximumBookingType($value)
    {
        if (!empty($value)) {
            $this->showMoreBookingFields = true;
        } else {
            $this->showMoreBookingFields = false;
            $this->maximum_booking_for = null;
            $this->maximum_bookings = null;
        }
    }

    public function updatedApprovalRequired($value)
    {
        $this->isApprovelRequired = $value == 1;
        if ($this->isApprovelRequired) {
            $this->users = User::whereDoesntHave('roles', function($query) {
                $query->where('name', 'student');
            })->get(['id', 'name', 'email']);
        } else {
            $this->approval_from = null;
            $this->users = [];
        }
    }

    public function submitTerms()
    {
        $this->validate();

        $data = [
            'instrument_id' => $this->instrument->id,
            'can_cancel_before_booking_slot' => $this->can_cancel_before_booking_slot,
            'approval_required' => $this->approval_required,
            'approval_from' => $this->approval_from,
            'sop' => $this->sop,
            'maximum_booking_type' => $this->maximum_booking_type,
            'maximum_bookings' => $this->maximum_bookings,
            'maximum_booking_for' => $this->maximum_booking_for,
            'maximum_sample_booking_per_day' => $this->maximum_sample_booking_per_day,
            'booking_slot_duration' => $this->booking_slot_duration ?? 30,
            'instrument_sample_questions' => !empty($this->instrument_sample_questions) ? json_encode($this->instrument_sample_questions) : null,
        ];

        if ($this->isEditing) {
            $this->instrumentTerm->update($data);
            $message = 'Instrument terms updated successfully.';
        } else {
            InstrumentTerm::create($data);
            $message = 'Instrument terms added successfully.';
        }

        $this->dispatch('toastSuccess', $message);
        $this->dispatch('hideInstrumentTermsForm');
    }

    public function showInstrumentTermsForm()
    {
        $this->dispatch('hideInstrumentTermsForm');
    }

    public function addQuestion()
    {
        $this->instrument_sample_questions[] = '';
    }

    public function removeQuestion($index)
    {
        unset($this->instrument_sample_questions[$index]);
        $this->instrument_sample_questions = array_values($this->instrument_sample_questions);
    }

    public function render()
    {
        return view('livewire.instruments.instrument-terms-form');
    }
}
