<?php

namespace App\Livewire\Instruments;

use App\Models\Instrument;
use App\Models\InstrumentService;
use Barryvdh\DomPDF\Facade\Pdf;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class InstrumentServiceRecord extends Component
{
    use WithPagination;
    use WithFileUploads;

    public $search = '';
    public $instrumentSearch = '';
    public $serviceType = '';
    public $status = 'completed';
    public $showCompleteModal = false;
    public $selectedService;
    public $description;
    public $cost;
    public $nextServiceDate = null;
    public $photos = [];
    public $serviceId;
    public $forSingleInstrument = false;
    public $showInstrumentServiceDetails = false;
    public $selectedInstrument;

    // Part Information
    public $parts = [];
    public $part_name = '';
    public $part_description = '';
    public $part_images = [];
    public $oldPartReceived = false;

    // Invoice Information
    public $service_paid_or_not = false;
    public $invoice_cost;
    public $invoice_date;
    public $invoice_image;

    public function updatingSearch()
    {
        $this->resetPage();
    }

    protected $listeners = [
        'hideInstrumentDetails' => 'handleHideInstrumentDetails',
    ];


    public function viewServiceDetail($id)
    {
        $this->showInstrumentServiceDetails = true;
        $this->selectedInstrument= $id;
    }

    public function handleHideInstrumentDetails()
    {
        $this->showInstrumentServiceDetails = false;
    }

    public function updatingInstrumentSearch()
    {
        $this->resetPage();
    }

    public function updatingServiceType()
    {
        $this->resetPage();
    }

    public function updatingStatus()
    {
        $this->resetPage();
    }

    public function markServiceDone($id)
    {
        $this->serviceId = $id;
        $this->selectedService = InstrumentService::find($id);
        $this->description = $this->selectedService->description;
        $this->cost = $this->selectedService->cost;
        $this->nextServiceDate = $this->selectedService->next_service_date;
        $this->serviceType = $this->selectedService->service_type;
        $this->showCompleteModal = true;
    }

    public function addMorePartInformation()
    {
        $this->validate([
            'part_name' => 'required',
            'part_description' => 'required',
            'part_images.*' => 'image|max:1024'
        ]);

        $partImagesPaths = [];
        if (!empty($this->part_images)) {
            foreach ($this->part_images as $image) {
                $partImagesPaths[] = $image->store('part-images', 'public');
            }
        }

        $this->parts[] = [
            'part_name' => $this->part_name,
            'part_description' => $this->part_description,
            'part_images' => $partImagesPaths,
            'oldPartReceived' => $this->oldPartReceived
        ];

        $this->reset(['part_name', 'part_description', 'part_images', 'oldPartReceived']);
    }

    public function completeService()
    {
        $this->validate([
            'serviceType' => 'required|in:repair,maintenance',
            'description' => 'required',
        ]);

        if ($this->service_paid_or_not) {
            $this->validate([
                'invoice_cost' => 'required|numeric',
                'invoice_date' => 'required|date',
                'invoice_image' => 'required|image|max:1024'
            ]);
        }

        $photosPaths = [];
        if (!empty($this->photos)) {
            foreach ($this->photos as $photo) {
                $photosPaths[] = $photo->store('service-photos', 'public');
            }
        }

        $invoiceDetails = null;
        if ($this->service_paid_or_not) {
            $invoiceImagePath = $this->invoice_image->store('invoice-images', 'public');
            $invoiceDetails = [
                'invoice_cost' => $this->invoice_cost,
                'invoice_date' => $this->invoice_date,
                'invoice_image' => $invoiceImagePath
            ];
        }

        $service = InstrumentService::find($this->serviceId);
        $service->update([
            'service_type' => $this->serviceType,
            'description' => $this->description,
            'next_service_date' => $this->nextServiceDate,
            'service_paid_or_not' => $this->service_paid_or_not,
            'instrument_details' => !empty($this->parts) ? json_encode($this->parts) : null,
            'invoice_details' => $invoiceDetails ? json_encode($invoiceDetails) : null,
            'status' => 'completed'
        ]);

        $instrument = Instrument::find($service->instrument_id);
        $instrument->update([
            'operating_status' => 'working'
        ]);

        $this->showCompleteModal = false;
        $this->reset(['description', 'cost', 'next_service_date', 'photos', 'serviceId', 'serviceType']);
        session()->flash('message', 'Service marked as completed successfully.');
    }

    public function exportToPdf()
    {
        $services = InstrumentService::query()
            ->when($this->forSingleInstrument && $this->instrument, function($query) {
                $query->where('instrument_id', $this->instrument->id);
            })
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('description', 'like', '%' . $this->search . '%')
                      ->orWhere('service_type', 'like', '%' . $this->search . '%');
                });
            })
            ->when(!$this->forSingleInstrument && $this->instrumentSearch, function($query) {
                $query->whereHas('instrument', function($q) {
                    $q->where('name', 'like', '%' . $this->instrumentSearch . '%');
                });
            })
            ->when($this->serviceType, function($query) {
                $query->where('service_type', $this->serviceType);
            })
            ->when($this->status !== 'All', function($query) {
                $query->where('status', $this->status);
            })
            ->with('instrument')
            ->get();

        $pdf = Pdf::loadView('exports.service-pdf', [
            'services' => $services
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'services.pdf');
    }

    public $instrument;

    public function render()
    {
        $services = InstrumentService::query()
            ->when($this->forSingleInstrument && $this->instrument, function($query) {
                $query->where('instrument_id', $this->instrument->id);
            })
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('description', 'like', '%' . $this->search . '%')
                      ->orWhere('service_type', 'like', '%' . $this->search . '%');
                });
            })
            ->when(!$this->forSingleInstrument && $this->instrumentSearch, function($query) {
                $query->whereHas('instrument', function($q) {
                    $q->where('name', 'like', '%' . $this->instrumentSearch . '%');
                });
            })
            ->when($this->serviceType, function($query) {
                $query->where('service_type', $this->serviceType);
            })
            ->when($this->status !== 'All', function($query) {
                $query->where('status', $this->status);
            })
            ->with('instrument')
            ->paginate(10);

        $instruments = $this->forSingleInstrument ? collect([$this->instrument]) : Instrument::all();

        return view('livewire.instruments.instrument-service-record', [
            'services' => $services,
            'instruments' => $instruments
        ]);
    }
}
