<?php

namespace App\Livewire\Instruments;

use App\Mail\InstrumentServiceRequestMail;
use App\Models\Instrument;
use App\Models\InstrumentAccessorie;
use App\Models\InstrumentPurchaseInfo;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Barryvdh\DomPDF\Facade\Pdf;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\InstrumentExport;

use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;
use Symfony\Component\HttpFoundation\StreamedResponse;



class InstrumentList extends Component
{
    use WithPagination;
    use WithFileUploads;

    public $showForm = false;
    public $search = '';
    public $status = 'All';
    public $isEditing = false;
    public $instrumentId = null;
    public $gridView = true; // Default to grid view
    public $viewInstrumentDetailView = false;
    public $viewInstrumentDetails;
    public $showInstrumentTermsForm = false;
    public $addTermsForInstrument;
    public $showQrCodeModal = false;
    public $showPricingModal = false;
    public $selectedInstrumentForPricing = null;
    public $showImageModal = false;
    public $selectedInstrumentForImage = null;
    public $qrCodeData;
    public $showStatusModal = false;
    public $selectedInstrumentId = null;
    public $selectedStatus = '';
    public $showAccessoryModal = false;
    public $selectedInstrument = null;
    public $accessoryName = '';
    public $accessoryModelNumber = '';
    public $accessoryPurchaseDate = '';
    public $accessoryPrice = '';
    public $accessoryDescription = '';
    public $accessoryStatus = 'available';
    public $accessoryPhoto = null;
    public $qrCodeCount = 1;
    public $qrFilename = '';
    public $showComplaintModal = false;
    public $selectedInstrumentForComplaint = null;
    public $complaintText = '';

    public $selectedLabId = null;

    public function mount()
    {
        $firstLab = \App\Models\Lab::first();
        if ($firstLab) {
            $this->selectedLabId = $firstLab->id;
        }
    }

    public function selectLab($labId)
    {
        $this->selectedLabId = $labId;
    }

    public function printQrCodes($instrumentId)
    {
        $this->validate([
            'qrCodeCount' => 'required|numeric|min:1|max:100',
        ]);

        $instrument = Instrument::findOrFail($instrumentId);
        $url = route('instrument.show', encode_id($instrumentId));

        // Generate QR code data for PDF
        $rendererStyle = new RendererStyle(400); // Smaller size for PDF
        $imageRenderer = new ImageRenderer($rendererStyle, new SvgImageBackEnd());
        $writer = new Writer($imageRenderer);
        $svgContent = $writer->writeString($url);

        $qrCodeData = 'data:image/svg+xml;base64,' . base64_encode($svgContent);

        // Generate PDF
        $pdf = PDF::loadView('livewire.instruments.qr-code-pdf', [
            'qrCodeData' => $qrCodeData,
            'count' => $this->qrCodeCount,
            'instrument' => $instrument,
        ]);

        $filename = "qr-codes-{$instrument->name}-{$instrument->serial_number}.pdf";
        $filename = Str::slug($filename);

        return response()->streamDownload(
            fn () => print($pdf->output()),
            $filename,
            [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="'.$filename.'"',
            ]
        );
    }

    public function generateQrCode($instrumentId)
    {
        $instrument = Instrument::findOrFail($instrumentId);

        // Generate URL with encoded ID
        $url = route('instrument.show', encode_id($instrumentId));

        // Generate QR code
        $rendererStyle = new RendererStyle(400); // Larger size for better scanning
        $imageRenderer = new ImageRenderer($rendererStyle, new SvgImageBackEnd());
        $writer = new Writer($imageRenderer);
        $svgContent = $writer->writeString($url);

        // Create download filename
        $filename = "qr-code-{$instrument->name}-{$instrument->serial_number}.svg";
        $filename = Str::slug($filename); // Sanitize filename

        $this->qrCodeData = 'data:image/svg+xml;base64,' . base64_encode($svgContent);
        $this->qrFilename = $filename;
        $this->instrument = $instrument;
        $this->showQrCodeModal = true;
    }

    protected $listeners = [
        'instrumentCreated' => 'handleInstrumentCreated',
        'instrumentUpdated' => 'handleInstrumentUpdated',
        'hideViewInstrument' => 'handleHideViewInstrument',
        'toastSuccess' => 'showSuccessToast',
        'hideInstrumentTermsForm' => 'handlehideInstrumentTermsForm'
    ];

    public function showSuccessToast($message)
    {
        session()->flash('success', $message);
    }

    public function toggleGridView()
    {
        $this->gridView = !$this->gridView;
    }

    public function handlehideInstrumentTermsForm()
    {
        $this->showInstrumentTermsForm = false;
    }

    public function hideForm()
    {
        $this->showForm = !$this->showForm;

        if (!$this->showForm) {
            $this->isEditing = false;
            $this->instrumentId = null;
            $this->dispatch('resetForm');
        }
    }

    public function editInstrument($id)
    {
        $this->instrumentId = $id;
        $this->isEditing = true;
        $this->showForm = true;
        $this->dispatch('editInstrument', instrumentId: $id);
    }

    public function viewInstrument($id)
    {
        $this->viewInstrumentDetails = Instrument::findOrFail($id);

        if ($this->viewInstrumentDetails) {
            $this->viewInstrumentDetailView = true;
        }
    }

    public function handleHideViewInstrument()
    {
        $this->viewInstrumentDetailView = false;
    }

    public function createInstrumentTerms($id)
    {
        $this->addTermsForInstrument = Instrument::with('instrumentTerm')->findOrFail($id);
        $this->showInstrumentTermsForm = true;
    }

    public function deleteInstrument($id)
    {
        // Find the Instrument and its purchase info
        $instrument = Instrument::findOrFail($id);
        $purchaseInfo = InstrumentPurchaseInfo::where('instrument_id', $id)->first();

        $bookings = $instrument->bookings;
        if ($bookings->count() > 0) {
            session()->flash('error', 'Cannot delete Instrument. It has associated bookings.');
            return;
        }

        // Delete files
        if ($instrument->photos) {
            foreach (json_decode($instrument->photos) as $photo) {
                if (Storage::disk('public')->exists($photo)) {
                    Storage::disk('public')->delete($photo);
                }
            }
        }

        if ($instrument->operational_manual && Storage::disk('public')->exists($instrument->operational_manual)) {
            Storage::disk('public')->delete($instrument->operational_manual);
        }

        if ($instrument->service_manual && Storage::disk('public')->exists($instrument->service_manual)) {
            Storage::disk('public')->delete($instrument->service_manual);
        }

        // Delete purchase info and instrument
        if ($purchaseInfo) {
            $purchaseInfo->delete();
        }
        $instrument->delete();

        session()->flash('success', 'Instrument successfully deleted.');
    }

    public function handleInstrumentCreated()
    {
        $this->showForm = false;
    }

    public function handleInstrumentUpdated()
    {
        $this->showForm = false;
        $this->isEditing = false;
        $this->instrumentId = null;
        session()->flash('success', 'Instrument updated successfully.');
    }

    public function updateStatus($id)
    {
        $instrument = Instrument::findOrFail($id);
        $this->selectedInstrumentId = $id; // Store only the ID instead of the whole model
        $this->selectedStatus = $instrument->operating_status;
        $this->showStatusModal = true;
    }

    public function updatePublicStatus($id)
    {
        $instrument = Instrument::findOrFail($id);
        $instrument->public_visibility_status = !$instrument->public_visibility_status;
        $instrument->update();
        session()->flash('success', 'Instrument public status updated successfully.');
    }

    public function sendForService($id)
    {
        $instrument = Instrument::findOrFail($id);
        $instrument->services()->create([
            'instrument_id' => $id,
            'service_type' => 'repair',
            'status' => 'pending',
        ]);

         Mail::to($instrument->engineer_email)->queue(new InstrumentServiceRequestMail($instrument));

        session()->flash('success', 'Instrument sent for service.');
    }

    public function confirmStatusUpdate()
    {
        $this->validate([
            'selectedStatus' => 'required|in:working,under_maintenance,calibration_required,faulty,retired'
        ]);

        $instrument = Instrument::findOrFail($this->selectedInstrumentId); // Fetch the instrument when needed
        $instrument->update([
            'operating_status' => $this->selectedStatus
        ]);

        $this->showStatusModal = false;
        $this->selectedInstrumentId = null; // Reset the ID instead of the model
        $this->selectedStatus = '';
        session()->flash('success', 'Status successfully updated.');
    }

    public function addAccessories($id)
    {
        $this->selectedInstrument = Instrument::findOrFail($id);
        $this->showAccessoryModal = true;
    }

    public function submitAccessory()
    {
        $this->validate([
            'accessoryName' => 'required|min:3',
            'accessoryModelNumber' => 'nullable',
            'accessoryPurchaseDate' => 'nullable|date|before:today',
            'accessoryPrice' => 'nullable|numeric',
            'accessoryDescription' => 'nullable|min:10',
            'accessoryStatus' => 'nullable|in:available,notAvailable',
            'accessoryPhoto' => 'nullable|image|max:1024',
        ]);


        $photoPath = null;

        if ($this->accessoryPhoto) {
            $photoPath = $this->accessoryPhoto->store('accessory-photos', 'public');
        }

        InstrumentAccessorie::create([
            'instrument_id' => $this->selectedInstrument->id,
            'name' => $this->accessoryName,
            'model_number' => $this->accessoryModelNumber,
            'purchase_date' => $this->accessoryPurchaseDate,
            'price' => $this->accessoryPrice,
            'description' => $this->accessoryDescription,
            'status' => $this->accessoryStatus,
            'photo' => $photoPath,
        ]);

        $this->showAccessoryModal = false;
        $this->selectedInstrument = null;
        $this->resetAccessoryFields();

        $this->dispatch('toastSuccess', 'Accessory has been added successfully.');
    }

    private function resetAccessoryFields()
    {
        $this->accessoryName = '';
        $this->accessoryModelNumber = '';
        $this->accessoryPurchaseDate = '';
        $this->accessoryPrice = '';
        $this->accessoryDescription = '';
        $this->accessoryStatus = 'available';
        $this->accessoryPhoto = null;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStatus()
    {
        $this->resetPage();
    }

    public function exportToPdf()
    {
        $query = $this->getFilteredQuery();
        $instruments = $query->get();

        $pdf = PDF::loadView('exports.instrument-pdf', [
            'instruments' => $instruments
        ]);

        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'instruments.pdf');
    }

    public function exportToExcel()
    {
        $query = $this->getFilteredQuery();
        return Excel::download(new InstrumentExport($query), 'instruments.xlsx');
    }

    private function getFilteredQuery()
    {
        $query = Instrument::with(['instrumentCategory', 'lab'])->latest();

        // Role-based filtering
        if (auth()->user()->hasRole('pi')) {
            $query->whereHas('lab', function($q) {
                $q->where('principal_investigator_id', auth()->user()->principalInvestigators->first()->id);
            });
        } elseif (auth()->user()->hasRole('technician')) {
            $query->where('technician_id', auth()->user()->technicians->first()->id);
        }
        // super_admin will see all instruments (no additional filtering needed)

        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('model_number', 'like', '%' . $this->search . '%')
                    ->orWhere('serial_number', 'like', '%' . $this->search . '%')
                    ->orWhere('operating_status', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->status !== 'All') {
            $query->where('operating_status', $this->status);
        }

        if ($this->selectedLabId) {
            $query->where('lab_id', $this->selectedLabId);
        }

        return $query;
    }

    public function toggleShowPricingModal($id)
    {
        $this->selectedInstrumentForPricing = Instrument::findOrFail($id);
        $this->showPricingModal = true;
    }

    public function hidePricingModal()
    {
        $this->showPricingModal = false;
        $this->selectedInstrumentForPricing = null;
    }

    public function toggleShowImageModal($id)
    {
        $this->selectedInstrumentForImage = Instrument::with(['purchaseInformation', 'lab', 'instrumentAccessories', 'instrumentTerm'])->findOrFail($id);
        $this->showImageModal = true;
    }

    public function hideImageModal()
    {
        $this->showImageModal = false;
        $this->selectedInstrumentForImage = null;
    }

    public function openComplaintModal($instrumentId)
    {
        $this->selectedInstrumentForComplaint = Instrument::findOrFail($instrumentId);
        $this->showComplaintModal = true;
        $this->complaintText = '';
    }

    public function submitOtherComplaint()
    {
        $this->validate([
            'complaintText' => 'required|string|min:10|max:1000',
        ]);

        if (!auth()->user()->hasRole('student') || !auth()->user()->students->first()) {
            session()->flash('error', 'Only students can submit complaints.');
            return;
        }

        \App\Models\OtherComplaint::create([
            'instrument_id' => $this->selectedInstrumentForComplaint->id,
            'student_id' => auth()->user()->students->first()->id,
            'complaint' => $this->complaintText,
            'status' => 'pending'
        ]);

        session()->flash('success', 'Complaint submitted successfully!');
        $this->showComplaintModal = false;
        $this->complaintText = '';
        $this->selectedInstrumentForComplaint = null;
    }

    public function render()
    {
        $query = $this->getFilteredQuery();

        if ($this->gridView) {
            $instruments = $query->get();
        } else {
            $instruments = $query->paginate(10);
        }

        $labs = \App\Models\Lab::orderBy('lab_name')->get();

        return view('livewire.instruments.instrument-list', [
            'instruments' => $instruments,
            'labs' => $labs,
            'isEditing' => $this->isEditing,
            'viewInstrumentDetails' => $this->viewInstrumentDetails,
        ]);
    }
}
