<?php

namespace App\Livewire\Instruments;

use App\Models\Instrument;
use App\Models\InstrumentCategory;
use App\Models\InstrumentPurchaseInfo;
use App\Models\Lab;
use App\Models\Technician;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\WithFileUploads;

class InstrumentForm extends Component
{
    use WithFileUploads;

    // Basic Information
    public $instrument_category;
    public $lab;
    public $technician;
    public $name;
    public $model_number;
    public $serial_number;
    public $description;
    public $operating_status = 'working';
    public $booking_type;
    public $slot_pricing_type = 'slot';
    public $booking_cost = [];

    // Photos and Documents
    public $photos = [];
    public $operational_manual;
    public $service_manual;
    public $video_link;

    // Purchase Information
    public $manufacturer_name;
    public $manufacturing_date;
    public $purchase_date;
    public $purchase_order_number;
    public $cost;
    public $funding_source;
    public $installation_date;
    public $engineer_name;
    public $engineer_email;
    public $engineer_mobile;
    public $engineer_address;

    public $isEditing = false;
    public $instrumentId = null;
    public $existingPhotos = [];
    public $existingOperationalManual;
    public $existingServiceManual;

    protected $listeners = [
        'editInstrument' => 'loadInstrumentData',
        'resetForm' => 'resetForm'
    ];

    protected function rules()
    {
        $rules = [
            'instrument_category' => 'required',
            'lab' => 'required',
            'technician' => 'nullable',
            'name' => 'required|string|max:255',
            'model_number' => 'nullable|string|max:255',
            'serial_number' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'operating_status' => 'nullable|string',
            'booking_type' => 'required|in:slot,sample,both',
            'slot_pricing_type' => 'required_if:booking_type,slot,both|in:slot,booking',
            'booking_cost' => 'required|array',
            'booking_cost.internal' => 'required|array',
            'booking_cost.other_educational_institute' => 'required|array',
            'booking_cost.r_and_d_labs' => 'required|array',
            'booking_cost.industries' => 'required|array',
            'booking_cost.startups' => 'required|array',
            'manufacturer_name' => 'nullable|string|max:255',
            'manufacturing_date' => 'nullable|date|before:today',
            'purchase_date' => ['nullable', 'date', 'before:today', 'after:manufacturing_date'],
            'purchase_order_number' => 'nullable|string|max:255',
            'cost' => 'nullable|integer|min:0',
            'funding_source' => 'nullable|string|max:255',
            'installation_date' => ['nullable', 'date', 'after:manufacturing_date', 'after_or_equal:purchase_date'],
            'video_link' => 'nullable|url',
            'engineer_name' => 'required|string|max:100',
            'engineer_email' => 'required|email|max:100',
            'engineer_mobile' => 'required|string|max:15|min:10',
            'engineer_address' => 'required|string|max:100'
        ];

        if (!$this->isEditing) {
            $rules['photos.*'] = 'required|image|max:2048';
            $rules['photos'] = 'required|array|min:1';
        } else {
            $rules['photos.*'] = 'nullable|image|max:2048';
            $rules['photos'] = 'required|array|min:1';
        }

        $rules['operational_manual'] = 'nullable|file|mimes:pdf|max:10240';
        $rules['service_manual'] =  'nullable|file|mimes:pdf|max:10240';

        return $rules;
    }

    public function loadInstrumentData($instrumentId)
    {
        $this->instrumentId = $instrumentId;
        $this->isEditing = true;

        $instrument = Instrument::findOrFail($instrumentId);
        $purchaseInfo = InstrumentPurchaseInfo::where('instrument_id', $instrumentId)->firstOrFail();

        // Load basic information
        $this->instrument_category = $instrument->instrument_category_id;
        $this->lab = $instrument->lab_id;
        $this->technician = $instrument->technician_id;
        $this->name = $instrument->name;
        $this->model_number = $instrument->model_number;
        $this->serial_number = $instrument->serial_number;
        $this->description = $instrument->description;
        $this->operating_status = $instrument->operating_status;
        $this->booking_type = $instrument->booking_type;
        $this->booking_cost = $instrument->booking_cost ?? [];
        $this->slot_pricing_type = isset($instrument->booking_cost['internal']['slot']) ? 'slot' : 'booking';
        $this->video_link = $instrument->video_link;
        $this->engineer_name = $instrument->engineer_name;
        $this->engineer_email = $instrument->engineer_email;
        $this->engineer_mobile = $instrument->engineer_mobile;
        $this->engineer_address = $instrument->engineer_address;

        // Load existing files
        $this->existingPhotos = json_decode($instrument->photos) ?? [];
        $this->existingOperationalManual = $instrument->operational_manual;
        $this->existingServiceManual = $instrument->service_manual;

        // Load purchase information
        $this->manufacturer_name = $purchaseInfo->manufacturer_name;
        $this->manufacturing_date = $purchaseInfo->manufacturing_date;
        $this->purchase_date = $purchaseInfo->purchase_date;
        $this->purchase_order_number = $purchaseInfo->purchase_order_number;
        $this->cost = $purchaseInfo->cost;
        $this->funding_source = $purchaseInfo->funding_source;
        $this->installation_date = $purchaseInfo->installation_date;
    }

    public function submit()
    {
        $this->validate();

        if ($this->isEditing) {
            $this->updateInstrument();
        } else {
            $this->createInstrument();
        }
    }

    protected function handleFileUploads()
    {
        $photosPaths = [];

        // Handle photos
        if ($this->photos) {
            foreach ($this->photos as $photo) {
                if (!is_string($photo)) {
                    $fileName = time() . '_' . rand(1000, 9999) . '_' . $photo->getClientOriginalName();
                    $path = $photo->storeAs('instrument_photos', $fileName, 'public');
                    $photosPaths[] = $path;
                }
            }
        }

        // Handle manuals
        $operationalManualPath = null;
        if ($this->operational_manual) {
            $fileName = time() . '_' . $this->operational_manual->getClientOriginalName();
            $operationalManualPath = $this->operational_manual->storeAs('instrument_manuals', $fileName, 'public');
        }

        $serviceManualPath = null;
        if ($this->service_manual) {
            $fileName = time() . '_' . $this->service_manual->getClientOriginalName();
            $serviceManualPath = $this->service_manual->storeAs('instrument_manuals', $fileName, 'public');
        }

        return [
            'photos' => $photosPaths,
            'operational_manual' => $operationalManualPath,
            'service_manual' => $serviceManualPath
        ];
    }

    protected function processBookingCost()
    {
        $processedBookingCost = [];
        $categories = ['internal', 'other_educational_institute', 'r_and_d_labs', 'industries', 'startups'];

        foreach ($categories as $category) {
            $processedBookingCost[$category] = [];

            if ($this->booking_type === 'sample') {
                // Only include sample pricing
                $processedBookingCost[$category]['sample'] = $this->booking_cost[$category]['sample'] ?? null;
                $processedBookingCost[$category]['sample_gst'] = $this->booking_cost[$category]['sample_gst'] ?? null;
            } elseif ($this->booking_type === 'slot') {
                // Only include slot/booking pricing based on slot_pricing_type
                $type = $this->slot_pricing_type;
                $processedBookingCost[$category][$type] = $this->booking_cost[$category][$type] ?? null;
                $processedBookingCost[$category][$type . '_gst'] = $this->booking_cost[$category][$type . '_gst'] ?? null;
            } elseif ($this->booking_type === 'both') {
                // Include sample and selected slot/booking pricing
                $processedBookingCost[$category]['sample'] = $this->booking_cost[$category]['sample'] ?? null;
                $processedBookingCost[$category]['sample_gst'] = $this->booking_cost[$category]['sample_gst'] ?? null;

                $type = $this->slot_pricing_type;
                $processedBookingCost[$category][$type] = $this->booking_cost[$category][$type] ?? null;
                $processedBookingCost[$category][$type . '_gst'] = $this->booking_cost[$category][$type . '_gst'] ?? null;
            }
        }

        return $processedBookingCost;
    }

    public function createInstrument()
    {
        $files = $this->handleFileUploads();

        // Create instrument
        $instrument = new Instrument();
        $instrument->instrument_category_id = $this->instrument_category;
        $instrument->lab_id = $this->lab;
        $instrument->technician_id = $this->technician;
        $instrument->name = $this->name;
        $instrument->model_number = $this->model_number;
        $instrument->serial_number = $this->serial_number;
        $instrument->description = $this->description;
        $instrument->operating_status = $this->operating_status;
        $instrument->booking_type = $this->booking_type;
        $instrument->booking_cost = $this->processBookingCost();

        $instrument->photos = json_encode($files['photos']);
        $instrument->operational_manual = $files['operational_manual'];
        $instrument->service_manual = $files['service_manual'];
        $instrument->video_link = $this->video_link;
        $instrument->engineer_name = $this->engineer_name;
        $instrument->engineer_email = $this->engineer_email;
        $instrument->engineer_mobile = $this->engineer_mobile;
        $instrument->engineer_address = $this->engineer_address;
        $instrument->save();

        // Create purchase info
        $purchaseInfo = new InstrumentPurchaseInfo();
        $purchaseInfo->instrument_id = $instrument->id;
        $purchaseInfo->manufacturer_name = $this->manufacturer_name;
        $purchaseInfo->manufacturing_date = $this->manufacturing_date;
        $purchaseInfo->purchase_date = $this->purchase_date;
        $purchaseInfo->purchase_order_number = $this->purchase_order_number;
        $purchaseInfo->cost = $this->cost;
        $purchaseInfo->funding_source = $this->funding_source;
        $purchaseInfo->installation_date = $this->installation_date;
        $purchaseInfo->save();

        $this->resetForm();
        $this->dispatch('toastSuccess', 'Instrument created successfully!');
        $this->dispatch('instrumentCreated');
    }

    public function updateInstrument()
    {
        $instrument = Instrument::findOrFail($this->instrumentId);
        $purchaseInfo = InstrumentPurchaseInfo::where('instrument_id', $this->instrumentId)->firstOrFail();

        $files = $this->handleFileUploads();

        // Update instrument
        $instrument->instrument_category_id = $this->instrument_category;
        $instrument->lab_id = $this->lab;
        $instrument->technician_id = $this->technician;
        $instrument->name = $this->name;
        $instrument->model_number = $this->model_number;
        $instrument->serial_number = $this->serial_number;
        $instrument->description = $this->description;
        $instrument->operating_status = $this->operating_status;
        $instrument->booking_type = $this->booking_type;
        $instrument->booking_cost = $this->processBookingCost();
        $instrument->engineer_name = $this->engineer_name;
        $instrument->engineer_email = $this->engineer_email;
        $instrument->engineer_mobile = $this->engineer_mobile;
        $instrument->engineer_address = $this->engineer_address;

        // Update files only if new ones are uploaded
        if (!empty($files['photos'])) {
            $currentPhotos = json_decode($instrument->photos, true) ?? [];

            // Combine existing and new photos
            $updatedPhotos = array_merge($currentPhotos, $files['photos']);
            $instrument->photos = json_encode($updatedPhotos);
        }

        if ($files['operational_manual']) {
            if ($instrument->operational_manual && Storage::disk('public')->exists($instrument->operational_manual)) {
                Storage::disk('public')->delete($instrument->operational_manual);
            }
            $instrument->operational_manual = $files['operational_manual'];
        }

        if ($files['service_manual']) {
            if ($instrument->service_manual && Storage::disk('public')->exists($instrument->service_manual)) {
                Storage::disk('public')->delete($instrument->service_manual);
            }
            $instrument->service_manual = $files['service_manual'];
        }

        $instrument->video_link = $this->video_link;
        $instrument->save();

        // Update purchase info
        $purchaseInfo->manufacturer_name = $this->manufacturer_name;
        $purchaseInfo->manufacturing_date = $this->manufacturing_date;
        $purchaseInfo->purchase_date = $this->purchase_date;
        $purchaseInfo->purchase_order_number = $this->purchase_order_number;
        $purchaseInfo->cost = $this->cost;
        $purchaseInfo->funding_source = $this->funding_source;
        $purchaseInfo->installation_date = $this->installation_date;
        $purchaseInfo->save();

        $this->resetForm();
        $this->dispatch('toastSuccess', 'Instrument updated successfully!');

        $this->dispatch('instrumentUpdated');
    }

    public function resetForm()
    {
        $this->reset([
            'instrument_category', 'lab', 'technician', 'name', 'model_number', 'serial_number',
            'description', 'operating_status', 'booking_cost', 'photos', 'operational_manual', 'service_manual',
            'video_link', 'manufacturer_name', 'manufacturing_date',
            'purchase_date', 'purchase_order_number', 'cost', 'funding_source',
            'installation_date','isEditing',
            'instrumentId', 'existingPhotos', 'existingOperationalManual', 'existingServiceManual',
            'engineer_name', 'engineer_email', 'engineer_mobile', 'engineer_address'
        ]);

        $this->operating_status = 'working';
    }

    public function render()
    {
        $instrumentCategories = InstrumentCategory::all();
        $labs = Lab::all();
        $technicians = Technician::where('status', 1)->get();
        return view('livewire.instruments.instrument-form', [
            'instrumentCategories' => $instrumentCategories,
            'labs' => $labs,
            'technicians' => $technicians
        ]);
    }
}
