<?php

namespace App\Livewire\Funds;

use App\Models\piFund;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\PiExport;
use Barryvdh\DomPDF\Facade\Pdf;

class FundsList extends Component
{
    use WithPagination;

    public $showAddFundsDiv = false;
    public $search = '';
    public $selectedFundId = null;

    protected $listeners = [
        'fundAdded' => 'handleFundAdded',
        'closeAddFundForm' => 'handleCloseAddFundForm',
        'fundUpdated' => 'handleFundUpdated'
    ];

    public function hideForm()
    {
        $this->showAddFundsDiv = !$this->showAddFundsDiv;
        $this->selectedFundId = null;
    }
    public function handleCloseAddFundForm()
    {
        $this->showAddFundsDiv = !$this->showAddFundsDiv;
        $this->selectedFundId = null;
    }

    public function editFund($fundId)
    {
        $this->selectedFundId = $fundId;
        $this->showAddFundsDiv = true;
    }

    public function handleFundAdded()
    {
        $this->showAddFundsDiv = false;
        session()->flash('success', 'Fund added successfully.');
    }
    public function handleFundUpdated()
    {
        $this->showAddFundsDiv = false;
        session()->flash('success', 'Fund updated successfully.');
    }

    public function exportToExcel()
    {
        $query = $this->getFilteredQuery();
        return Excel::download(new PiExport($query), 'pi_funds.xlsx');
    }

    public function exportToPdf()
    {
        $query = $this->getFilteredQuery();
        $funds = $query->get();
        $pdf = PDF::loadView('exports.pi-funds-pdf', ['funds' => $funds]);
        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'pi_funds.pdf');
    }

    private function getFilteredQuery()
    {
        $query = piFund::with('principalInvestigator')->latest();

        if ($this->search) {
            $query->whereHas('principalInvestigator', function($q) {
                $q->where('first_name', 'like', '%' . $this->search . '%')
                  ->orWhere('last_name', 'like', '%' . $this->search . '%');
            });
        }

        return $query;
    }

    public function render()
    {
        $query = $this->getFilteredQuery();
        $funds = $query->paginate(10);

        return view('livewire.funds.funds-list', [
            'funds' => $funds
        ]);
    }
}
