<?php

namespace App\Livewire\Bookings;

use App\Models\Booking;
use App\Models\bookingsForApproval;
use App\Models\Instrument;
use App\Models\Student;
use App\Models\Slot;
use Carbon\Carbon;
use Livewire\Component;
use Livewire\WithPagination;

class PendingForApproval extends Component
{
    use WithPagination;

    public $search = '';
    public $studentSearch = '';
    public $instrumentSearch = '';
    public $status = 'All';
    public $showCancelModal = false;
    public $selectedBookingId = null;
    public $cancellationRemark = '';
    public $studentView = false;
    public $showUpdateStatusModel = false;
    public $remark;
    public $currentBookingStatus;


    public function mount()
    {
        $this->studentView = auth()->user()->hasRole('student');
    }

    public function updateBookingStatus($bookingId)
    {
        $this->selectedBookingId = $bookingId;
        $this->showUpdateStatusModel = true;
    }

    public function updateStatus()
    {
        $this->validate([
            'currentBookingStatus' => 'required|in:approve,reject',
            'remark' => 'required|string|max:255'
        ]);

        try {
            \DB::transaction(function () {
                $bookingApproval = bookingsForApproval::findOrFail($this->selectedBookingId);
                $booking = Booking::findOrFail($bookingApproval->booking_id);

                if ($this->currentBookingStatus === 'approve') {

                    $bookingCost = $booking->booking_cost;

                    $pi = $booking->student->principalInvestigator;
                    $pi->update([
                        'available_funds' => $pi->available_funds - $bookingCost
                    ]);

                    // Update booking status
                    $booking->update(['status' => 'confirmed']);
                    $booking->update(['description' => $this->remark]);
                    $bookingApproval->update([
                        'status' => 'approved',
                        'remark' => $this->remark,
                        'approved_by' => auth()->user()->name
                    ]);

                    session()->flash('success', 'Booking approved successfully!');
                } else {
                    // Update booking status for rejection
                    $booking->update(['status' => 'cancelled']);
                    $booking->update(['description' => $this->remark]);
                    $bookingApproval->update([
                        'status' => 'rejected',
                        'remark' => $this->remark,
                        'approved_by' => auth()->user()->name
                    ]);

                    session()->flash('success', 'Booking rejected successfully!');
                }
            });

            $this->reset(['selectedBookingId', 'remark', 'currentBookingStatus']);
            $this->showUpdateStatusModel = false;
        } catch (\Exception $e) {
            session()->flash('error', 'An error occurred while updating the booking status.');
        }
    }



    public function render()
    {
        $query = bookingsForApproval::query()
            ->with(['booking.student', 'booking.instrument', 'booking.slots']);

        // Filter based on user role
        if (!auth()->user()->hasRole('super_admin')) {
            $query->where('user_id', auth()->user()->id);
        }

        // Apply search filters
        if ($this->search) {
            $query->whereHas('booking', function ($q) {
                $q->whereHas('student', function ($q) {
                    $q->where('first_name', 'like', '%' . $this->search . '%')
                        ->orWhere('last_name', 'like', '%' . $this->search . '%');
                })->orWhereHas('instrument', function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }

        if ($this->studentSearch) {
            $query->whereHas('booking.student', function ($q) {
                $q->where('first_name', 'like', '%' . $this->studentSearch . '%');
            });
        }

        if ($this->instrumentSearch) {
            $query->whereHas('booking.instrument', function ($q) {
                $q->where('name', 'like', '%' . $this->instrumentSearch . '%');
            });
        }

        if ($this->status !== 'All') {
            $query->where('status', $this->status);
        }

        $bookings = $query->latest()->paginate(10);
        $students = Student::orderBy('first_name')->get();
        $instruments = Instrument::orderBy('name')->get();

        return view('livewire.bookings.pending-for-approval', [
            'bookings' => $bookings,
            'students' => $students,
            'instruments' => $instruments,
        ]);
    }
}
