<?php
namespace App\Livewire\Bookings;

use App\Models\bookingsForApproval;
use App\Models\Instrument;
use App\Models\NotifyForInstrument;
use App\Models\Student;
use App\Models\Slot;
use App\Models\Booking;
use App\Models\InstrumentTerm;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Carbon\Carbon;

class Create extends Component
{
    public $isEditing = false;
    public $student;
    public $instrument;
    public $date;
    public $selectedSlots = [];
    public $description;
    public $sopAcknowledged = false;
    public $booking_type = 'slot';
    public $sample_answers = [];
    public $sample_quantity = 1;

    public $selectedInstrument = null;
    public $selectedDate = null;
    public $allSlots = [];
    public $bookedSlotIds = [];
    public $passedSlotIds = [];
    public $slotBookingDetails = [];
    public $isInstrumentAvailable = true;
    public $instrumentTerms = null;
    public $showSlotsSection = true;
    public $showDateField = true;
    public $showNotifyButton = false;

    protected $listeners = [
        'showForm' => 'handleShowForm',
        'toastError' => 'showErrorToast',
        'toastSuccess' => 'showSuccessToast'
    ];

    public function showSuccessToast($message)
    {
        session()->flash('success', $message);
    }

    public function showErrorToast($message)
    {
        session()->flash('error', $message);
    }

    public function handleShowForm()
    {
        $this->isEditing = true;
    }

    public function mount($instrument_id = null, $student_id = null)
    {
        $this->date = Carbon::today()->format('Y-m-d');

        // If parameters are provided, pre-fill the form and show it
        if ($instrument_id && $student_id) {
            $this->instrument = $instrument_id;
            $this->student = $student_id;
            $this->isEditing = true;

            // Trigger the instrument selection logic
            $this->updatedInstrument($instrument_id);
        }
    }

    public function hideForm()
    {
        $this->reset(['student', 'instrument', 'selectedSlots', 'selectedInstrument', 'description', 'sopAcknowledged', 'instrumentTerms', 'booking_type', 'sample_answers']);
        $this->date = Carbon::today()->format('Y-m-d');
        $this->selectedDate = null;
        $this->bookedSlotIds = [];
        $this->passedSlotIds = [];
        $this->slotBookingDetails = [];
        $this->isEditing = false;
    }

    public function updatedInstrument($instrumentId)
    {
        $this->selectedInstrument = Instrument::find($instrumentId);
        $this->reset(['selectedSlots', 'sopAcknowledged']); // Reset selected slot and SOP acknowledgment when instrument changes


        if ($this->selectedInstrument) {
            $this->instrumentTerms = InstrumentTerm::where('instrument_id', $instrumentId)->first();
        } else {
            $this->instrumentTerms = null;
        }

        $this->booking_type = $this->selectedInstrument->booking_type;
        $this->showSlotsSection = $this->booking_type === 'slot';


        if ($this->selectedInstrument->operating_status === 'under_maintenance') {
            $this->selectedDate = null;
            $this->isInstrumentAvailable = false;
            $this->allSlots = [];
        } else {
            $this->showDateField = true;
            $this->isInstrumentAvailable = true;
            $this->selectedDate = Carbon::today()->format('d-m-Y'); // Set selected date to today
            $this->loadAllSlots();
        }
    }

    public function notifyForInstrument()
    {
        $this->validate([
            'student' => 'required|exists:students,id',
            'instrument' => 'required|exists:instruments,id'
        ]);

        try {
            // Check if notification already exists
            $existingNotification = NotifyForInstrument::where('student_id', $this->student)
                ->where('instrument_id', $this->instrument)
                ->first();

            if ($existingNotification) {
                session()->flash('error', 'You are already in the notification list for this instrument.');
                return;
            }

            // Create notification
            NotifyForInstrument::create([
                'student_id' => $this->student,
                'instrument_id' => $this->instrument
            ]);

            session()->flash('success', 'You will be notified when the instrument becomes available.');
            $this->hideForm();

        } catch (\Exception $e) {
            session()->flash('error', 'An error occurred while setting up the notification. Please try again.');
        }
    }

    public function updatedDate($date)
    {
        if (Carbon::parse($date)->isBefore(Carbon::today())) {
            $this->date = Carbon::today()->format('Y-m-d');
            session()->flash('error', 'Cannot select past dates.');
            return;
        }

        if ($this->selectedInstrument) {
            $this->selectedDate = Carbon::parse($date)->format('Y-m-d');
            $this->selectedSlots = []; // Reset selected slots
            $this->loadAllSlots();
        }
    }

    public function selectSlot($slotId)
    {
        if (in_array($slotId, $this->bookedSlotIds) || in_array($slotId, $this->passedSlotIds)) {
            return;
        }

        if (in_array($slotId, $this->selectedSlots)) {
            // Deselect the slot and its duration group
            $this->deselectSlotGroup($slotId);
        } else {
            // Select the slot with duration-based auto-selection
            $this->selectSlotWithDuration($slotId);
        }
    }

    protected function selectSlotWithDuration($slotId)
    {
        // Get booking slot duration from instrument terms (default 30 minutes)
        $bookingDuration = $this->instrumentTerms && $this->instrumentTerms->booking_slot_duration 
            ? $this->instrumentTerms->booking_slot_duration 
            : 30;

        // Calculate how many 30-minute slots are needed
        $slotsNeeded = $bookingDuration / 30;

        // Get the selected slot
        $selectedSlot = Slot::find($slotId);
        if (!$selectedSlot) {
            return;
        }

        // Get all slots ordered by start time
        $allSlots = $this->allSlots->sortBy('start_time');
        
        // Find the index of the selected slot
        $selectedIndex = $allSlots->search(function($slot) use ($slotId) {
            return $slot->id == $slotId;
        });

        if ($selectedIndex === false) {
            return;
        }

        // Get consecutive slots starting from the selected slot
        $slotsToSelect = [];
        for ($i = 0; $i < $slotsNeeded; $i++) {
            $slotIndex = $selectedIndex + $i;
            if ($slotIndex >= $allSlots->count()) {
                session()->flash('error', 'Not enough consecutive slots available for the required duration.');
                return;
            }

            $slot = $allSlots->values()[$slotIndex];
            
            // Check if slot is available
            if (in_array($slot->id, $this->bookedSlotIds) || in_array($slot->id, $this->passedSlotIds)) {
                session()->flash('error', 'Some required slots are not available for the booking duration.');
                return;
            }

            $slotsToSelect[] = $slot->id;
        }

        // Verify slots are consecutive (no time gaps)
        if (!$this->areSlotConsecutive($slotsToSelect)) {
            session()->flash('error', 'Required slots are not consecutive in time.');
            return;
        }

        // Add the slots to selection
        foreach ($slotsToSelect as $slotToSelect) {
            if (!in_array($slotToSelect, $this->selectedSlots)) {
                $this->selectedSlots[] = $slotToSelect;
            }
        }
    }

    protected function deselectSlotGroup($slotId)
    {
        // Get booking slot duration from instrument terms (default 30 minutes)
        $bookingDuration = $this->instrumentTerms && $this->instrumentTerms->booking_slot_duration 
            ? $this->instrumentTerms->booking_slot_duration 
            : 30;

        // Calculate how many 30-minute slots are needed for minimum booking
        $slotsNeeded = $bookingDuration / 30;

        // Find which consecutive group this slot belongs to
        $groups = $this->groupConsecutiveSlots($this->selectedSlots);

        // Find which group contains the clicked slot
        foreach ($groups as $group) {
            if (in_array($slotId, $group)) {
                // If the group has exactly the minimum required slots, remove the entire group
                // If the group has more than minimum slots, remove only the minimum duration from the clicked position
                if (count($group) === $slotsNeeded) {
                    // Remove entire group
                    $this->selectedSlots = array_diff($this->selectedSlots, $group);
                } else {
                    // Find the position of clicked slot in the group
                    $clickedIndex = array_search($slotId, $group);
                    
                    // Determine which minimum-duration block to remove
                    $blockStartIndex = intval($clickedIndex / $slotsNeeded) * $slotsNeeded;
                    $slotsToRemove = array_slice($group, $blockStartIndex, $slotsNeeded);
                    
                    // Remove the block
                    $this->selectedSlots = array_diff($this->selectedSlots, $slotsToRemove);
                }
                break;
            }
        }
    }

    protected function areSlotConsecutive($slotIds)
    {
        if (count($slotIds) <= 1) {
            return true;
        }

        $slots = Slot::whereIn('id', $slotIds)->orderBy('start_time')->get();
        
        for ($i = 1; $i < $slots->count(); $i++) {
            $prevSlot = $slots[$i - 1];
            $currentSlot = $slots[$i];
            
            // Check if end time of previous slot equals start time of current slot
            if ($prevSlot->end_time !== $currentSlot->start_time) {
                return false;
            }
        }
        
        return true;
    }

    protected function groupConsecutiveSlots($selectedSlots)
    {
        if (empty($selectedSlots)) {
            return [];
        }

        sort($selectedSlots);
        $groups = [];
        $currentGroup = [];
        
        $allSlots = $this->allSlots->sortBy('start_time')->keyBy('id');
        
        foreach ($selectedSlots as $slotId) {
            if (empty($currentGroup)) {
                $currentGroup = [$slotId];
            } else {
                // Check if this slot is consecutive to the last slot in current group
                $lastSlotInGroup = end($currentGroup);
                $lastSlot = $allSlots[$lastSlotInGroup];
                $currentSlot = $allSlots[$slotId];
                
                if ($lastSlot->end_time === $currentSlot->start_time) {
                    // This slot is consecutive, add to current group
                    $currentGroup[] = $slotId;
                } else {
                    // This slot is not consecutive, start a new group
                    if (count($currentGroup) > 0) {
                        $groups[] = $currentGroup;
                    }
                    $currentGroup = [$slotId];
                }
            }
        }
        
        // Add the last group
        if (count($currentGroup) > 0) {
            $groups[] = $currentGroup;
        }
        
        return $groups;
    }

    public function clearAllSelections()
    {
        $this->selectedSlots = [];
    }



    protected function loadAllSlots()
    {
        if (!$this->selectedInstrument || !$this->date) {
            return;
        }

        $this->allSlots = Slot::orderBy('start_time')->get();
        $selectedDate = Carbon::parse($this->date);

        // Reset arrays
        $this->bookedSlotIds = [];
        $this->passedSlotIds = [];
        $this->slotBookingDetails = [];

        // Get booked slots with booking details
        $bookings = Booking::where('instrument_id', $this->selectedInstrument->id)
            ->where('date', $selectedDate->format('Y-m-d'))
            ->whereIn('status', ['confirmed', 'pending'])
            ->with(['slots', 'student.user'])
            ->get();

        foreach ($bookings as $booking) {
            foreach ($booking->slots as $slot) {
                $this->bookedSlotIds[] = $slot->id;
                $this->slotBookingDetails[$slot->id] = [
                    'booked_by' => $booking->student->first_name . ' ' . $booking->student->last_name,
                    'booking_id' => $booking->id
                ];
            }
        }

        // Handle expired/past slots - separate from booked slots
        if ($selectedDate->isToday()) {
            $this->allSlots->each(function ($slot) use ($selectedDate) {
                // Handle midnight time properly: 24:00:00 means end of current day (midnight)
                if ($slot->end_time === '24:00:00') {
                    // For 24:00:00, create datetime for end of current day (23:59:59) and add 1 second
                    $slotDateTime = Carbon::parse($selectedDate->format('Y-m-d') . ' 23:59:59')->addSecond();
                } else {
                    // For all other times, parse normally
                    $slotDateTime = Carbon::parse($selectedDate->format('Y-m-d') . ' ' . $slot->end_time);
                }

                if ($slotDateTime->isPast() && !in_array($slot->id, $this->bookedSlotIds)) {
                    $this->passedSlotIds[] = $slot->id;
                }
            });
        }
        elseif ($selectedDate->isPast()) {
            // For past dates, all unbooked slots are passed
            $this->passedSlotIds = $this->allSlots->pluck('id')->diff($this->bookedSlotIds)->toArray();
        }
    }

    public function rules()
    {
        $rules = [
            'student' => 'required|exists:students,id',
            'instrument' => 'required|exists:instruments,id',
            'booking_type' => 'required|in:slot,sample',
            'date' => 'required|date|after_or_equal:today',
            'description' => 'nullable|string|max:1000'
        ];

        if ($this->booking_type === 'slot') {
            $rules['selectedSlots'] = 'required|array|min:1';
            $rules['selectedSlots.*'] = 'exists:slots,id';
        } else {
            $rules['sample_quantity'] = 'required|integer|min:1';
        }


        if ($this->instrumentTerms && $this->instrumentTerms->sop) {
            $rules['sopAcknowledged'] = 'accepted';
        }

        return $rules;
    }

    public function submit()
    {
        $this->validate();

        // Only validate and process slots if booking type is 'slot'
        if ($this->booking_type === 'slot') {
            // Check if any selected slot is booked or passed
            $bookedSlots = array_intersect($this->selectedSlots, $this->bookedSlotIds);
            $passedSlots = array_intersect($this->selectedSlots, $this->passedSlotIds);

            if (count($bookedSlots) > 0) {
                session()->flash('error', 'One or more selected slots have been booked. Please select different slots.');
                return;
            }

            if (count($passedSlots) > 0) {
                session()->flash('error', 'One or more selected slots have passed. Please select different slots.');
                return;
            }

            // Validate slot groups based on booking duration
            $bookingDuration = $this->instrumentTerms && $this->instrumentTerms->booking_slot_duration 
                ? $this->instrumentTerms->booking_slot_duration 
                : 30;
            $slotsNeeded = $bookingDuration / 30;

            // Group consecutive slots
            $slotGroups = $this->groupConsecutiveSlots($this->selectedSlots);
            
            // Validate each group meets minimum duration requirement
            foreach ($slotGroups as $group) {
                if (count($group) < $slotsNeeded) {
                    session()->flash('error', 'Invalid slot selection. Each booking requires at least ' . $slotsNeeded . ' consecutive slots (' . $bookingDuration . ' minutes minimum).');
                    return;
                }
                
                if (!$this->areSlotConsecutive($group)) {
                    session()->flash('error', 'Selected slots must be consecutive in time.');
                    return;
                }
            }
        }

        try {
            // Get student's PI and check booking limits
            $student = Student::find($this->student);
            $pi = $student->principalInvestigator;
            $selectedDate = Carbon::parse($this->date);

            // Get instrument terms
            $instrumentTerms = InstrumentTerm::where('instrument_id', $this->instrument)->first();

            if ($instrumentTerms && $instrumentTerms->maximum_booking_type && $instrumentTerms->maximum_bookings) {
                $startDate = null;
                $endDate = null;

                // Set date range based on booking type
                switch ($instrumentTerms->maximum_booking_type) {
                    case 'day':
                        $startDate = $selectedDate->copy()->startOfDay();
                        $endDate = $selectedDate->copy()->endOfDay();
                        break;
                    case 'week':
                        $startDate = $selectedDate->copy()->startOfWeek();
                        $endDate = $selectedDate->copy()->endOfWeek();
                        break;
                    case 'month':
                        $startDate = $selectedDate->copy()->startOfMonth();
                        $endDate = $selectedDate->copy()->endOfMonth();
                        break;
                }

                // Count existing bookings for this PI and their students
                $existingBookings = Booking::whereHas('student', function($query) use ($pi) {
                    $query->where('principal_investigator_id', $pi->id);
                })
                    ->where('instrument_id', $this->instrument)
                    ->whereBetween('date', [$startDate->format('Y-m-d'), $endDate->format('Y-m-d')])
                    ->where('status', 'confirmed')
                    ->count();

                if ($existingBookings >= $instrumentTerms->maximum_bookings) {
                    session()->flash('error', "Booking limit reached for this {$instrumentTerms->maximum_booking_type}. Maximum {$instrumentTerms->maximum_bookings} bookings allowed.");
                    return;
                }
            }

            // Calculate booking cost
            $instrument = Instrument::find($this->instrument);
            $booking_costs = $instrument->booking_cost;
            $bookingCost = 0;
            $user_type = auth()->user()->user_type;

            if ($this->booking_type === 'sample') {
                // For sample booking type
                $sample_cost = $booking_costs[$user_type]['sample'] ?? 0;
                $sample_gst = $booking_costs[$user_type]['sample_gst'] ?? 0;
                $bookingCost = $sample_cost * $this->sample_quantity;
                // Add GST
                $bookingCost = $bookingCost + ($bookingCost * ($sample_gst / 100));
            } else {
                // For slot booking type
                // Check if booking cost has 'booking' key for per-booking pricing
                if (isset($booking_costs[$user_type]['booking'])) {
                    $slot_cost = $booking_costs[$user_type]['booking'];
                    $slot_gst = $booking_costs[$user_type]['booking_gst'] ?? 0;
                    $bookingCost = $slot_cost;
                } else {
                    // Per slot pricing
                    $slot_cost = $booking_costs[$user_type]['slot'] ?? 0;
                    $slot_gst = $booking_costs[$user_type]['slot_gst'] ?? 0;
                    $bookingCost = $slot_cost * count($this->selectedSlots);
                }
                // Add GST
                $bookingCost = $bookingCost + ($bookingCost * ($slot_gst / 100));
            }

            DB::transaction(function () use ($bookingCost, $pi, $instrumentTerms) {
                // Check if approval is required
                $status = $instrumentTerms && $instrumentTerms->approval_required ? 'pending' : 'confirmed';

                if ($this->booking_type === 'slot') {
                    // Group consecutive slots and create separate bookings for each group
                    $slotGroups = $this->groupConsecutiveSlots($this->selectedSlots);
                    
                    $totalCost = 0;
                    
                    foreach ($slotGroups as $groupIndex => $slotGroup) {
                        // Calculate cost per booking group
                        $groupCost = 0;
                        $user_type = auth()->user()->user_type;
                        
                        if (isset($booking_costs[$user_type]['booking'])) {
                            // Per-booking pricing
                            $slot_cost = $booking_costs[$user_type]['booking'];
                            $slot_gst = $booking_costs[$user_type]['booking_gst'] ?? 0;
                            $groupCost = $slot_cost;
                        } else {
                            // Per slot pricing
                            $slot_cost = $booking_costs[$user_type]['slot'] ?? 0;
                            $slot_gst = $booking_costs[$user_type]['slot_gst'] ?? 0;
                            $groupCost = $slot_cost * count($slotGroup);
                        }
                        // Add GST
                        $groupCost = $groupCost + ($groupCost * ($slot_gst / 100));
                        $totalCost += $groupCost;

                        $booking = Booking::create([
                            'student_id' => $this->student,
                            'instrument_id' => $this->instrument,
                            'date' => Carbon::parse($this->date)->format('Y-m-d'),
                            'description' => $this->description . (count($slotGroups) > 1 ? ' (Booking ' . ($groupIndex + 1) . ' of ' . count($slotGroups) . ')' : ''),
                            'status' => $status,
                            'booking_type' => $this->booking_type,
                            'booked_by' => auth()->user()->user_type,
                            'booking_cost' => $groupCost,
                            'sample_quantity' => null
                        ]);

                        // Attach slots for this group
                        $booking->slots()->attach($slotGroup);

                        // If approval is required, create booking approval request
                        if ($status === 'pending') {
                            bookingsForApproval::create([
                                'booking_id' => $booking->id,
                                'user_id' => $instrumentTerms->approval_from,
                                'status' => 'pending'
                            ]);
                        }
                    }
                    
                    // Update PI's available funds with total cost only if booking is confirmed and booked_by is internal
                    if ($status === 'confirmed' && auth()->user()->user_type === 'internal') {
                        $pi->update([
                            'available_funds' => $pi->available_funds - $totalCost
                        ]);
                    }
                } else {
                    // Sample booking - single booking
                    $booking = Booking::create([
                        'student_id' => $this->student,
                        'instrument_id' => $this->instrument,
                        'date' => Carbon::parse($this->date)->format('Y-m-d'),
                        'description' => $this->description,
                        'status' => $status,
                        'booking_type' => $this->booking_type,
                        'booked_by' => auth()->user()->user_type,
                        'booking_cost' => $bookingCost,
                        'sample_quantity' => $this->sample_quantity
                    ]);

                    // If approval is required, create booking approval request
                    if ($status === 'pending') {
                        bookingsForApproval::create([
                            'booking_id' => $booking->id,
                            'user_id' => $instrumentTerms->approval_from,
                            'status' => 'pending'
                        ]);
                    } else {
                        // Update PI's available funds only if booking is confirmed and booked_by is internal
                        if (auth()->user()->user_type === 'internal') {
                            $pi->update([
                                'available_funds' => $pi->available_funds - $bookingCost
                            ]);
                        }
                    }
                }
            });

            // Reset form
            $this->reset(['student', 'instrument', 'selectedSlots', 'selectedInstrument', 'description', 'sopAcknowledged', 'instrumentTerms']);
            $this->date = Carbon::today()->format('Y-m-d');
            $this->selectedDate = null;
            $this->allSlots = [];
            $this->bookedSlotIds = [];
            $this->passedSlotIds = [];
            $this->slotBookingDetails = [];

            // Show appropriate success message
            if ($instrumentTerms && $instrumentTerms->approval_required) {
                session()->flash('error', 'Your booking is sent for approval. You will be notified once the booking got confirmed.');
            } else {
                session()->flash('success', 'Booking created successfully!');
            }
            $this->hideForm();

        } catch (\Exception $e) {
            Log::error('Booking creation failed: ' . $e->getMessage(), [
                'exception' => $e,
            ]);
            session()->flash('error', 'An error occurred while creating the booking. Please try again.');
        }
    }

    public function render()
    {
        // if role is super_admin then show all students, if role is pi then show only students under that pi and if role is student then show only that student
        if (auth()->user()->hasRole('super_admin')) {
            $students = Student::orderBy('first_name')->get();
        } elseif (auth()->user()->hasRole('pi')) {
            $students = Student::where('principal_investigator_id', auth()->user()->principalInvestigators->first()->id)->orderBy('first_name')->get();
        }
        elseif (auth()->user()->hasRole('technician')) {
            $students = Student::orderBy('first_name')->get();
        }
        else {
            $students = Student::where('id', auth()->user()->students->first()->id)->orderBy('first_name')->get();
            $this->student = $students->first()->id;
        }

        return view('livewire.bookings.create', [
            'students' => $students,
            'instruments' => Instrument::whereIn('operating_status', ['working', 'under_maintenance'])->orderBy('name')->get(),
        ]);
    }
}
