<?php

namespace App\Livewire\Bookings;

use App\Models\Booking;
use App\Models\Student;
use App\Models\Instrument;
use App\Models\InstrumentComplaint;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\BookingExport;

class BookingList extends Component
{
    use WithPagination;
    use WithFileUploads;

    public $search = '';
    public $studentSearch = '';
    public $instrumentSearch = '';
    public $status = 'All';
    public $studentView = false;

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStudentSearch()
    {
        $this->resetPage();
    }

    public function updatingInstrumentSearch()
    {
        $this->resetPage();
    }

    public function updatingStatus()
    {
        $this->resetPage();
    }

    public function showForm()
    {
        $this->dispatch('showForm');
    }

    public function exportToExcel()
    {
        $query = Booking::query()->with(['student', 'instrument', 'slot'])->latest();
        return Excel::download(new BookingExport($query), 'bookings.xlsx');
    }

    public function exportToPdf()
    {
        $query = Booking::query()->with(['student', 'instrument', 'slot'])->latest();
        $bookings = $query->get();
        $pdf = Pdf::loadView('exports.booking-pdf', ['bookings' => $bookings]);
        return response()->streamDownload(function() use ($pdf) {
            echo $pdf->output();
        }, 'bookings.pdf');
    }

    public $showCancelModal = false;
    public $selectedBooking = null;
    public $cancellationRemark = '';

    // Complaint Modal Properties
    public $showComplaintModal = false;
    public $complaintSubject = '';
    public $complaintDescription = '';
    public $complaintImage = null;
    public $complaintBooking = null;

    public function cancelBooking($bookingId)
    {
        $this->selectedBooking = Booking::with(['slots', 'instrument.instrumentTerm'])->find($bookingId);

        // Get cancellation time limit from instrument terms
        $cancelBeforeHours = 0; // Default value
        if ($this->selectedBooking->instrument->instrumentTerm &&
            $this->selectedBooking->instrument->instrumentTerm->can_cancel_before_booking_slot) {
            $cancelBeforeHours = $this->selectedBooking->instrument->instrumentTerm->can_cancel_before_booking_slot;
        }

        // Calculate cancellation deadline based on booking type
        if ($this->selectedBooking->booking_type === 'sample') {
            // For sample bookings, use the booking date
            $bookingDateTime = $this->selectedBooking->date . ' 00:00:00';
            $cancelBeforeTime = date('Y-m-d H:i:s', strtotime($bookingDateTime . ' -' . $cancelBeforeHours . ' hours'));
        } else {
            // For slot bookings, use the slot start time
            $bookingDateTime = $this->selectedBooking->date . ' ' . $this->selectedBooking->slots->first()->start_time;
            $cancelBeforeTime = date('Y-m-d H:i:s', strtotime($bookingDateTime . ' -' . $cancelBeforeHours . ' hours'));
        }

        if (now() >= $cancelBeforeTime) {
            $this->dispatch('toastError', 'Cannot cancel booking within ' . $cancelBeforeHours . ' hours of the ' .
                ($this->selectedBooking->booking_type === 'sample' ? 'booking date' : 'slot time') . '.');
            return;
        }

        $this->showCancelModal = true;
    }

    public function confirmCancellation()
    {
        $this->validate([
            'cancellationRemark' => 'required',
        ]);

        \DB::transaction(function () {
            // Calculate booking cost to refund
            $booking = $this->selectedBooking;
            $bookingCost = $this->selectedBooking->booking_cost;

            // Get student's PI and update available funds
            $student = $booking->student;
            $pi = $student->principalInvestigator;

            $this->selectedBooking->update([
                'status' => 'cancelled',
                'description' => $this->cancellationRemark
            ]);

            if ($pi->available_funds<0)
                $pi->update([
                    'available_funds' => $pi->available_funds + $bookingCost
                ]);
            else{
                $pi->update([
                    'available_funds' => $pi->available_funds - $bookingCost
                ]);
            }
        });

        $this->showCancelModal = false;
        $this->selectedBooking = null;
        $this->cancellationRemark = '';
        $this->dispatch('toastSuccess', 'Booking has been cancelled successfully.');
    }

    public function raiseComplaint($bookingId)
    {
        $this->complaintBooking = Booking::with(['slots', 'instrument'])->find($bookingId);

        // Check if a complaint already exists for this booking
        $existingComplaint = InstrumentComplaint::where('booking_id', $bookingId)->first();
        if ($existingComplaint) {
            $this->dispatch('toastError', 'A complaint has already been registered for this booking.');
            return;
        }

        // For slot-based bookings, check the time window
        if ($this->complaintBooking->booking_type === 'slot') {
            $bookingDateTime = $this->complaintBooking->date . ' ' . $this->complaintBooking->slots->first()->start_time;
            $slotStartTime = strtotime($bookingDateTime);
            $fifteenMinutesAfterStart = strtotime('+15 minutes', $slotStartTime);

            if (time() < $slotStartTime || time() > $fifteenMinutesAfterStart) {
                $this->dispatch('toastError', 'Complaints for slot-based bookings can only be raised within 15 minutes of slot start time.');
                return;
            }
        }

        $this->showComplaintModal = true;
    }

    public function submitComplaint()
    {
        $this->validate([
            'complaintSubject' => 'required|min:5',
            'complaintDescription' => 'required|min:10',
            'complaintImage' => 'nullable|image|max:1024',
        ]);

        $imagePath = null;
        if ($this->complaintImage) {
            $imagePath = $this->complaintImage->store('complaint-images', 'public');
        }

        InstrumentComplaint::create([
            'instrument_id' => $this->complaintBooking->instrument_id,
            'student_id' => $this->complaintBooking->student_id,
            'booking_id' => $this->complaintBooking->id,
            'subject' => $this->complaintSubject,
            'description' => $this->complaintDescription,
            'image' => $imagePath,
            'status' => 'pending'
        ]);

        $this->showComplaintModal = false;
        $this->complaintBooking = null;
        $this->complaintSubject = '';
        $this->complaintDescription = '';
        $this->complaintImage = null;

        $this->dispatch('toastSuccess', 'Complaint has been submitted successfully.');
    }

    public function render()
    {


        if (auth()->user()->hasRole('super_admin')) {
            $query = Booking::query()->with(['student', 'instrument', 'slots'])->latest();
        } elseif (auth()->user()->hasRole('pi')) {
            $query = Booking::whereHas('student', function($q) {
                $q->where('principal_investigator_id', auth()->user()->principalInvestigators->first()->id);
            })->with(['student', 'instrument', 'slots'])->latest();
        } elseif (auth()->user()->hasRole('technician')) {
            $query = Booking::whereHas('instrument', function($q) {
                $q->where('technician_id', auth()->user()->technicians->first()->id);
            })->with(['student', 'instrument', 'slots'])->latest();
        } else {
            $this->studentView = true;
            $query = Booking::where('student_id', auth()->user()->students->first()->id)->with(['student', 'instrument', 'slots'])->latest();
        }


        // Apply general search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->whereHas('student', function($q) {
                    $q->where('first_name', 'like', '%' . $this->search . '%')
                      ->orWhere('last_name', 'like', '%' . $this->search . '%');
                })
                ->orWhereHas('instrument', function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }

        // Apply student-specific search
        if ($this->studentSearch) {
            $query->whereHas('student', function($q) {
                $q->where('first_name', 'like', '%' . $this->studentSearch . '%')
                  ->orWhere('last_name', 'like', '%' . $this->studentSearch . '%');
            });
        }

        // Apply instrument-specific search
        if ($this->instrumentSearch) {
            $query->whereHas('instrument', function($q) {
                $q->where('name', 'like', '%' . $this->instrumentSearch . '%');
            });
        }

        // Apply status filter
        if ($this->status !== 'All') {
            $query->where('status', $this->status);
        }


        return view('livewire.bookings.booking-list', [
            'bookings' => $query->paginate(10),
            'students' => Student::orderBy('first_name')->get(),
            'instruments' => Instrument::orderBy('name')->get()
        ]);
    }
}
