<?php

namespace App\Helpers;

use InvalidArgumentException;

class UnitConverter
{
    /**
     * Unit conversion definitions with base units and conversion factors
     * Base units: ml for liquids, mg for solids
     */
    private static array $conversions = [
        // Liquid units (base: ml)
        'ml' => ['base' => 'ml', 'factor' => 1, 'type' => 'liquid'],
        'L' => ['base' => 'ml', 'factor' => 1000, 'type' => 'liquid'],
        
        // Solid units (base: mg)
        'mg' => ['base' => 'mg', 'factor' => 1, 'type' => 'solid'],
        'g' => ['base' => 'mg', 'factor' => 1000, 'type' => 'solid'],
        'kg' => ['base' => 'mg', 'factor' => 1000000, 'type' => 'solid'],
    ];

    /**
     * Convert a value from any unit to its base unit
     * 
     * @param float $value The value to convert
     * @param string $unit The unit to convert from
     * @return float The value in base units
     * @throws InvalidArgumentException If unit is not supported
     */
    public static function convertToBase(float $value, string $unit): float
    {
        if (!self::isValidUnit($unit)) {
            throw new InvalidArgumentException("Unsupported unit: {$unit}");
        }

        if ($value < 0) {
            throw new InvalidArgumentException("Value cannot be negative");
        }

        return $value * self::$conversions[$unit]['factor'];
    }

    /**
     * Convert a value from base unit to any supported unit
     * 
     * @param float $value The value in base units
     * @param string $unit The unit to convert to
     * @return float The value in the specified unit
     * @throws InvalidArgumentException If unit is not supported
     */
    public static function convertFromBase(float $value, string $unit): float
    {
        if (!self::isValidUnit($unit)) {
            throw new InvalidArgumentException("Unsupported unit: {$unit}");
        }

        if ($value < 0) {
            throw new InvalidArgumentException("Value cannot be negative");
        }

        return $value / self::$conversions[$unit]['factor'];
    }

    /**
     * Convert between any two supported units
     * 
     * @param float $value The value to convert
     * @param string $fromUnit The unit to convert from
     * @param string $toUnit The unit to convert to
     * @return float The converted value
     * @throws InvalidArgumentException If units are not supported or incompatible
     */
    public static function convert(float $value, string $fromUnit, string $toUnit): float
    {
        if (!self::isValidUnit($fromUnit)) {
            throw new InvalidArgumentException("Unsupported source unit: {$fromUnit}");
        }

        if (!self::isValidUnit($toUnit)) {
            throw new InvalidArgumentException("Unsupported target unit: {$toUnit}");
        }

        if (!self::areUnitsCompatible($fromUnit, $toUnit)) {
            throw new InvalidArgumentException("Cannot convert between {$fromUnit} and {$toUnit} - incompatible unit types");
        }

        if ($value < 0) {
            throw new InvalidArgumentException("Value cannot be negative");
        }

        // Convert to base unit first, then to target unit
        $baseValue = self::convertToBase($value, $fromUnit);
        return self::convertFromBase($baseValue, $toUnit);
    }

    /**
     * Check if a unit is valid/supported
     * 
     * @param string $unit The unit to check
     * @return bool True if unit is supported
     */
    public static function isValidUnit(string $unit): bool
    {
        return array_key_exists($unit, self::$conversions);
    }

    /**
     * Check if two units are compatible (same type: liquid or solid)
     * 
     * @param string $unit1 First unit
     * @param string $unit2 Second unit
     * @return bool True if units are compatible
     */
    public static function areUnitsCompatible(string $unit1, string $unit2): bool
    {
        if (!self::isValidUnit($unit1) || !self::isValidUnit($unit2)) {
            return false;
        }

        return self::$conversions[$unit1]['type'] === self::$conversions[$unit2]['type'];
    }

    /**
     * Get the base unit for a given unit
     * 
     * @param string $unit The unit to get base for
     * @return string The base unit
     * @throws InvalidArgumentException If unit is not supported
     */
    public static function getBaseUnit(string $unit): string
    {
        if (!self::isValidUnit($unit)) {
            throw new InvalidArgumentException("Unsupported unit: {$unit}");
        }

        return self::$conversions[$unit]['base'];
    }

    /**
     * Get the unit type (liquid or solid)
     * 
     * @param string $unit The unit to get type for
     * @return string The unit type
     * @throws InvalidArgumentException If unit is not supported
     */
    public static function getUnitType(string $unit): string
    {
        if (!self::isValidUnit($unit)) {
            throw new InvalidArgumentException("Unsupported unit: {$unit}");
        }

        return self::$conversions[$unit]['type'];
    }

    /**
     * Get all supported units
     * 
     * @return array Array of supported units
     */
    public static function getSupportedUnits(): array
    {
        return array_keys(self::$conversions);
    }

    /**
     * Get units by type (liquid or solid)
     * 
     * @param string $type The type to filter by ('liquid' or 'solid')
     * @return array Array of units of the specified type
     * @throws InvalidArgumentException If type is not valid
     */
    public static function getUnitsByType(string $type): array
    {
        if (!in_array($type, ['liquid', 'solid'])) {
            throw new InvalidArgumentException("Invalid unit type: {$type}. Must be 'liquid' or 'solid'");
        }

        return array_keys(array_filter(self::$conversions, function($conversion) use ($type) {
            return $conversion['type'] === $type;
        }));
    }

    /**
     * Format a value with its unit for display
     * 
     * @param float $value The value to format
     * @param string $unit The unit
     * @param int $decimals Number of decimal places
     * @return string Formatted string
     * @throws InvalidArgumentException If unit is not supported
     */
    public static function formatWithUnit(float $value, string $unit, int $decimals = 2): string
    {
        if (!self::isValidUnit($unit)) {
            throw new InvalidArgumentException("Unsupported unit: {$unit}");
        }

        return number_format($value, $decimals) . ' ' . $unit;
    }

    /**
     * Get the best unit for display (automatically choose appropriate unit)
     * For example, 1500ml becomes 1.5L, 2000mg becomes 2g
     * 
     * @param float $baseValue Value in base units
     * @param string $baseUnit The base unit (ml or mg)
     * @return array ['value' => converted_value, 'unit' => best_unit]
     * @throws InvalidArgumentException If base unit is not valid
     */
    public static function getBestDisplayUnit(float $baseValue, string $baseUnit): array
    {
        if (!in_array($baseUnit, ['ml', 'mg'])) {
            throw new InvalidArgumentException("Base unit must be 'ml' or 'mg'");
        }

        $type = self::getUnitType($baseUnit);
        $units = self::getUnitsByType($type);
        
        // Sort units by factor (largest first)
        usort($units, function($a, $b) {
            return self::$conversions[$b]['factor'] <=> self::$conversions[$a]['factor'];
        });

        // Find the best unit (largest unit where value >= 1)
        foreach ($units as $unit) {
            $convertedValue = self::convertFromBase($baseValue, $unit);
            if ($convertedValue >= 1) {
                return [
                    'value' => $convertedValue,
                    'unit' => $unit
                ];
            }
        }

        // If no suitable unit found, use the base unit
        return [
            'value' => $baseValue,
            'unit' => $baseUnit
        ];
    }

    /**
     * Format value with the best display unit
     * 
     * @param float $baseValue Value in base units
     * @param string $baseUnit The base unit (ml or mg)
     * @param int $decimals Number of decimal places
     * @return string Formatted string with best unit
     */
    public static function formatBestUnit(float $baseValue, string $baseUnit, int $decimals = 2): string
    {
        $best = self::getBestDisplayUnit($baseValue, $baseUnit);
        return self::formatWithUnit($best['value'], $best['unit'], $decimals);
    }
}