<?php

namespace App\Helpers;

use Carbon\Carbon;

class InventoryValidationHelper
{
    /**
     * Check if an item is low stock
     */
    public static function isLowStock($currentStock, $minStockLevel)
    {
        if (!$minStockLevel) {
            return false;
        }
        
        return $currentStock <= $minStockLevel;
    }

    /**
     * Check if an item is out of stock
     */
    public static function isOutOfStock($currentStock)
    {
        return $currentStock <= 0;
    }

    /**
     * Check if an item is expired
     */
    public static function isExpired($expiryDate)
    {
        if (!$expiryDate) {
            return false;
        }

        return Carbon::parse($expiryDate)->isPast();
    }

    /**
     * Check if an item is expiring soon
     */
    public static function isExpiringSoon($expiryDate, $days = 30)
    {
        if (!$expiryDate) {
            return false;
        }

        $expiry = Carbon::parse($expiryDate);
        return $expiry->isFuture() && $expiry->diffInDays(now()) <= $days;
    }

    /**
     * Get stock status with appropriate styling classes
     */
    public static function getStockStatusClasses($currentStock, $minStockLevel = null)
    {
        if (self::isOutOfStock($currentStock)) {
            return [
                'badge' => 'bg-red-100 text-red-800',
                'icon' => 'fa-times-circle',
                'text' => 'Out of Stock'
            ];
        }

        if (self::isLowStock($currentStock, $minStockLevel)) {
            return [
                'badge' => 'bg-yellow-100 text-yellow-800',
                'icon' => 'fa-exclamation-triangle',
                'text' => 'Low Stock'
            ];
        }

        return [
            'badge' => 'bg-green-100 text-green-800',
            'icon' => 'fa-check-circle',
            'text' => 'In Stock'
        ];
    }

    /**
     * Get expiry status with appropriate styling classes
     */
    public static function getExpiryStatusClasses($expiryDate)
    {
        if (!$expiryDate) {
            return [
                'badge' => 'bg-gray-100 text-gray-800',
                'icon' => 'fa-info-circle',
                'text' => 'No Expiry'
            ];
        }

        if (self::isExpired($expiryDate)) {
            return [
                'badge' => 'bg-red-100 text-red-800',
                'icon' => 'fa-exclamation-triangle',
                'text' => 'Expired'
            ];
        }

        if (self::isExpiringSoon($expiryDate)) {
            return [
                'badge' => 'bg-yellow-100 text-yellow-800',
                'icon' => 'fa-clock',
                'text' => 'Expiring Soon'
            ];
        }

        return [
            'badge' => 'bg-green-100 text-green-800',
            'icon' => 'fa-check',
            'text' => 'Valid'
        ];
    }

    /**
     * Format unit display name
     */
    public static function formatUnitName($unit)
    {
        $units = [
            'ml' => 'Milliliters',
            'L' => 'Liters',
            'mg' => 'Milligrams',
            'g' => 'Grams',
            'kg' => 'Kilograms'
        ];

        return $units[$unit] ?? ucfirst($unit);
    }

    /**
     * Get unit category (liquid or solid)
     */
    public static function getUnitCategory($unit)
    {
        $liquidUnits = ['ml', 'L'];
        return in_array($unit, $liquidUnits) ? 'liquid' : 'solid';
    }
}