<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class LowStockReportExport implements FromQuery, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $query;

    public function __construct($query)
    {
        $this->query = $query;
    }

    public function query()
    {
        return $this->query;
    }

    public function headings(): array
    {
        return [
            'Item Name',
            'Category',
            'Item Type',
            'Current Stock',
            'Minimum Stock Level',
            'Stock Deficit',
            'Stock Status',
            'Last Restocked',
            'Storage Conditions'
        ];
    }

    public function map($item): array
    {
        $currentStock = $item->getCurrentStock();
        $deficit = max(0, $item->min_stock_level - $currentStock);
        
        $status = 'Low Stock';
        if ($currentStock <= 0) {
            $status = 'Out of Stock';
        } elseif ($currentStock <= ($item->min_stock_level * 0.5)) {
            $status = 'Critical';
        }

        $lastRestocked = $item->stockBatches()
            ->where('transaction_type', 'in')
            ->latest('received_date')
            ->first();

        return [
            $item->name,
            $item->category->name ?? 'N/A',
            ucfirst($item->item_type ?? 'regular'),
            $currentStock,
            $item->min_stock_level ?? 0,
            $deficit,
            $status,
            $lastRestocked ? $lastRestocked->received_date->format('Y-m-d') : 'Never',
            $item->storage_conditions ?? 'N/A'
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
}