<?php

namespace App\Console\Commands;

use App\Models\StockAlert;
use App\Models\InventoryItem;
use App\Models\StockBatch;
use Illuminate\Console\Command;

class GenerateStockAlerts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:generate-alerts {--days=30 : Days ahead to check for expiry alerts}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate stock alerts for low stock levels and expiring items';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Generating stock alerts...');
        
        $daysAhead = $this->option('days');
        
        // Generate low stock alerts
        $lowStockCount = $this->generateLowStockAlerts();
        
        // Generate expiry alerts
        $expiryCount = $this->generateExpiryAlerts($daysAhead);
        
        $this->info("Generated {$lowStockCount} low stock alerts");
        $this->info("Generated {$expiryCount} expiry alerts");
        $this->info('Stock alert generation completed successfully!');
        
        return 0;
    }

    private function generateLowStockAlerts(): int
    {
        $alertsGenerated = 0;
        
        $items = InventoryItem::whereNotNull('min_stock_level')
            ->where('min_stock_level', '>', 0)
            ->get();

        foreach ($items as $item) {
            $currentStock = $item->getCurrentStock();
            
            if ($currentStock <= $item->min_stock_level) {
                // Check if alert already exists and is unresolved
                $existingAlert = StockAlert::where('inventory_item_id', $item->id)
                    ->where('alert_type', 'low_stock')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    StockAlert::create([
                        'inventory_item_id' => $item->id,
                        'alert_type' => 'low_stock',
                        'alert_message' => "Low stock alert: {$item->name} has {$currentStock} units remaining (minimum: {$item->min_stock_level})",
                        'resolved' => false,
                    ]);
                    
                    $alertsGenerated++;
                    $this->line("  - Low stock alert created for: {$item->name}");
                }
            }
        }
        
        return $alertsGenerated;
    }

    private function generateExpiryAlerts(int $daysAhead): int
    {
        $alertsGenerated = 0;
        
        $batches = StockBatch::whereNotNull('expiry_date')
            ->where('current_quantity', '>', 0)
            ->get();

        foreach ($batches as $batch) {
            $daysUntilExpiry = now()->diffInDays($batch->expiry_date, false);
            
            // Create expiry alerts for items expiring within the specified timeframe
            if ($daysUntilExpiry <= $daysAhead && $daysUntilExpiry >= 0) {
                // Check if alert already exists and is unresolved
                $existingAlert = StockAlert::where('batch_id', $batch->id)
                    ->where('alert_type', 'expiry')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    $message = $daysUntilExpiry == 0 
                        ? "Expiry alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expires today!"
                        : "Expiry alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expires in {$daysUntilExpiry} days";

                    StockAlert::create([
                        'inventory_item_id' => $batch->inventory_item_id,
                        'batch_id' => $batch->id,
                        'alert_type' => 'expiry',
                        'alert_message' => $message,
                        'resolved' => false,
                    ]);
                    
                    $alertsGenerated++;
                    $this->line("  - Expiry alert created for: {$batch->inventoryItem->name} (Batch: {$batch->batch_number})");
                }
            }
            
            // Create expired alerts for items that have already expired
            if ($daysUntilExpiry < 0) {
                $existingAlert = StockAlert::where('batch_id', $batch->id)
                    ->where('alert_type', 'expiry')
                    ->where('resolved', false)
                    ->first();

                if (!$existingAlert) {
                    $daysExpired = abs($daysUntilExpiry);
                    StockAlert::create([
                        'inventory_item_id' => $batch->inventory_item_id,
                        'batch_id' => $batch->id,
                        'alert_type' => 'expiry',
                        'alert_message' => "Expired alert: {$batch->inventoryItem->name} (Batch: {$batch->batch_number}) expired {$daysExpired} days ago",
                        'resolved' => false,
                    ]);
                    
                    $alertsGenerated++;
                    $this->line("  - Expired alert created for: {$batch->inventoryItem->name} (Batch: {$batch->batch_number})");
                }
            }
        }
        
        return $alertsGenerated;
    }
}
